			INCLUDE "code/macro.asm"	;ALIGN macro definition
			INCLUDE "code/struct.asm"	;Data structures for cars and cameras
			INCLUDE "variables.lst"		;Variable addresses

			; MSX BIOS Routines
WRTVDP			EQU	00047h			; Write B to VDP register C
INIGRP			EQU	00072h			; Initialize VDP to Graphics Mode
GICINI			EQU	00090h			; Initializes PSG and sets initial value for the PLAY statemen
ERAFNK			EQU	000CCh			; Erase function key display
ENASLT     		EQU	00024h      		; enable slot
RSLREG     		EQU	00138h      		; read primary slot select register


			; MSX System variables
BIOSVDPREG              EQU     0F3DFh                  ; VDP's reg 0
CLIKSW  		EQU     0F3DBh  		; Keyboard sound
EXPTBL     		EQU	0FCC1h			; slot is expanded or not
VDPREG			EQU	0FF00h			; New VDP reg value location

			; RAM buffers
screenbuffer		EQU	0C000h			;768 bytes buffer for screens
spritesbuffer		EQU	0C300h			;144 bytes buffer for sprites, some extra variables after it
mapa_tablecolors	EQU	mapa+3072+512+2
keyreadbuffer		EQU	mapa+3072+520
theredefkey		EQU	mapa+3072+540		;Store read keys while redefining
mapa_checks		EQU	mapa+4096
variables		EQU	0D600h			;Variables zone

			ORG	0


;********
;INSROM0: The first 16k block starts here. At the beginning, this block isn't paged in.
;********

;******
;rst00: Called by RST instruction in just one byte, this routine sets RAM in page 2
;******
rst00:			di			;Disable interrupts
			push	hl
			push	af
			ld	hl,a8ramslot	;Get RAM slot
			jr	switchSLOT	;Keep paging

;******
;rst08: Called by RST instruction in just one byte, this routine sets the INSROM in page 2
;******
rst08:			di			;Disable interrupts
			push	hl
			push	af
			ld	hl,a8romslot	;Get ROM slot
switchSLOT:		push	de
			ld	e,(hl)		;slot in E
			dec	l
			ld	d,(hl)		;subslot in D
			dec	l
			ld	a,(hl)		;preslot in A
			out	(#a8),a		;Page the slot needed to select page 2 subslot. Memory can't be accessed now.
			ld	a,(65535)
			cpl			;Get INSROM2 subslot
			and	207		;Clear page 2 bits
			or	d		;Form subslot
			ld	(65535),a	;Select subslots.
			ld	a,e
			out	(#a8),a		;Select slot. Memory can be accessed now again.
			pop	de
			pop	af
			pop	hl
			ei			;Enable interrupts
			ret

;************
;int_routine: Executed every frame, at address 56 (38h) in mode 1.
;************
			ORG	56
int_routine:		push	iy			;Save registers
			push	ix
			push	hl
			push	de
			push	bc
			push	af
			ex	af,af'
			push	af
			ld	a,(VDPportREAD)		;Get VDP read port
			ld	c,a
			in	a,(c)			;Clear VDP interrupt flag
			and	a
			jp	p,int_end		;Exit if the interrupt wasn't generated by VDP
			ld	hl,int_ticks
			inc	(hl)			;Increase tick counter
			ld	hl,(interrupt_redir)	;Get interrupt routine
			jp	(hl)			;Jump to it

int_end:		pop	af			;Restore registers
			ex	af,af'
			pop	af
			pop	bc
			pop	de
			pop	hl			
			pop	ix
			pop	iy
                	ei                              ; Reactivate interrupts	
			ret				; Return to program already


;*****************
;(included stuff): Lots of things
;*****************
			INCLUDE	"code/deexo.asm"	;Exomizer decompression routine
			INCLUDE	"code/dump.asm"		;Code to dump the screen
			INCLUDE	"wyzplayer.asm"		;WYZPlayer to play ingame & interlude PSG music

colorsets:		
			INCLUDE	"data/colorsets.asm"	;Colorsets
			ALIGN	1024
charset:		
			INCBIN  "bindata/charset1.bin"	;Charset (00-95) aligned to 1024
			INCBIN  "bindata/charset2.bin"	;Extra characters
mcolorspacked:
			INCLUDE	"data/mcolors.asm"	;Marker Colors Packed
			INCLUDE	"data/background.asm"	;Background tiles data
			INCLUDE	"data/tablas.asm"	;Several static data tablas
			INCLUDE "data/spd_table.asm"	;Speed table
			INCLUDE "data/messages.asm"	;Messages
			INCLUDE	"data/velocidad.asm"	;Velocimeter graphics
			INCLUDE "data/car_tiles.asm"	;Car tiles graphics
			INCLUDE	"data/car_colors.asm"	;Car colors data
			INCLUDE	"data/car_graph.asm"	;Car frames made of tiles
			INCLUDE "data/spriteheight.asm"	;Sprite vertical fine position
			INCLUDE	"data/championships.asm";Championships data
tilesunpacked:		
			INCLUDE	"data/tiles.asm"	;Common Tiles Patterns
colorsunpacked:		
			INCLUDE	"data/color.asm"	;Common Tiles Colors

;*******************
;getkey_personalize: Wait for keys RETURN, BACKSPACE or 0-8 to be pressed. If RETURN is pressed, exit with carry set, otherwise exit with the value of key in B (9 for BACKSPACE).
;*******************
getkey_personalize:	call	get_key				;Wait for a keypress, get scan and row
			call	wait_nokey			;Wait until no key is being pressed
			ex	af,af'				;Row in A, scan in A'
			and	15				;Clear high bits
			cp	7				;Check if we're in row 7
			jr	z,row_7				
			cp	1				;Check if we're in row 1 or 0
			jr	c,row_0
			jr	nz,getkey_personalize		;No row 0,1 or 7: keep trying
								;Row 1
			ex	af,af'				;A = scan
			ld	b,8				;return value
			rrca					;Get bit 0
			ret	nc				;If it's 0, key 8 was pressed: exit
			jr	getkey_personalize		;Otherwise, try again

row_7:			ex	af,af'				;A = scan
			rlca					;check RETURN bit
			ccf					;complement carry, so we exit with carry enabled
			ret	c				;exit with RETURN
			ld	b,9				;return value for Backspace
			rlca
			rlca					;check BACKSPACE bit
			ret	nc				;If it's 0, BACKSPACE was pressed: exit
			jr	getkey_personalize		;Otherwise, try again

row_0:			ex	af,af'				;A=scan
			ld	b,-1				;initial return value-1
getkey_keeprotating:	inc	b				;next return value
			rrca					;check bit
			jr	c,getkey_keeprotating		;keep rotating until the key has been found
			ret					;(code should never reach this position)

;***************
;printmicromap2: Print the micromap addressed by A, in position E
;***************
printmicromap2:		inc	a			;1 step minimum
			ld	bc,64			;Map lenght
			ld	hl,mapa-64		;First map -1 address
pmm2_getmapaddr:	add	hl,bc			;Next map
			dec	a
			jr	nz,pmm2_getmapaddr	;Get the map position
			ex	de,hl			;DE = map position in memory
			ld	h,0			;HL = VRAM character to be written
			add	hl,hl
			add	hl,hl
			add	hl,hl			;HL= VRAM position to be written (character*8)
			push	hl
			push	de
			ld	bc,128			;Number of bytes
			call	ldrvrm			;Copy bytes to VRAM in first third
			pop	de
			pop	hl
			ld	a,h
			add	a,8
			ld	h,a			;Same address now in second third
			ld	bc,128			;Number of bytes
			jp	ldrvrm			;Copy bytes to VRAM in second third, then return

;*************
;attrblock4x4:
;*************
attrblock4x4:		ld	(mes_attrblockcol),a	;Set color in block message
			ld	a,e
			ld	(mes_attrblockpos),a	;Set position in block message
			ld	bc,mes_attrblock
			call	printmessage_nr		;Print block message
			ld	hl,mes_attrblock
			ld	a,(hl)
			xor	1
			ld	(hl),a			;Switch third in message between 1 and 2
			ld	b,h
			ld	c,l
			jp	printmessage_nr		;Print block message again, then return

;*******************
;putmiddlemicromaps: Print the micromaps in the middle of the championship personalization screen
;*******************
pmidmicro_error:	dec	l			;Point to previous track
			inc	a
			ld	(hl),a			;Set 1 lap
			jr	pmidmicro_putattr	;Go back to painting loop

putmiddlemicromaps:	ld	b,8			;Number of maps
			ld	hl,championship_personal+6	;Personalized data, number of laps of first track
			ld	e,128			;character 128 for the first micromap
							;128,132,136,140
							;129,133,137,141
							;-----------------third change
							;130,134,138,142
							;131,135,139,143

pmidmicro_putattr:	ld	a,(hl)			;Get number of laps for the current track
			inc	l			;Point to next track
			and	a			;Check if it's 0
			jr	z,pmidmicro_blackhere
			ld	a,113			;Non zero: Cyan color
			push	bc			;Keep registers
			push	de
			push	hl
			call	attrblock4x4		;Paint cyan attributes for map and number of tracks
			pop	hl			;Restore registers
			pop	de
			pop	bc
			ld	a,e
			add	a,16			;Point to next character
			ld	e,a
			djnz	pmidmicro_putattr	;Do it for all tracks
			ld	a,8			;All maps!
			ld	(person_maps),a		;Set number of personal maps
			ld	e,240			;Point to last map
			ld	a,209			;Magenta
			push	hl
			call	attrblock4x4		;Selected map is magenta
			pop	hl
			jr	pmidmicro_continue	;Continue
pmidmicro_blackhere:	ld	a,8
			sub	b			;A = number of tracks
			jr	z,pmidmicro_error	;It shouldn't have a 0, minimum is 1, so fix it
			ld	(person_maps),a		;Keep it

			ld	a,221			;Magenta ink and paper, to clear stage
			push	bc
			push	de
			push	hl
			call	attrblock4x4		;Set a magenta block
			pop	hl
			pop	de
			pop	bc
			dec	b			;Decrease counter
			jr	z,pmidmicro_continue	;If we were in the last map, no need to do anything else
pmidmicro_blackem:	ld	a,e
			add	a,16			;Point to next track characters
			ld	e,a
			xor	a			;Black color
			push	bc
			push	de
			push	hl
			call	attrblock4x4		;Paint a black block
			pop	hl
			pop	de
			pop	bc
			djnz	pmidmicro_blackem	;Do it for all remaining positions

pmidmicro_continue:	ld	e,128			;Character position to start printing the tracks
			ld	l,(championship_personal+14)%256	;Point to tracks*2
			ld	b,8			;8 tracks
pmidmicro_putmapsup:	push	bc
			ld	a,(hl)			;Get track number*2
			push	de
			push	hl
			call	printmicromap2		;Paint track
			pop	hl
			inc	l			;Point to next track number
			pop	de
			ld	a,e
			add	a,16
			ld	e,a			;Point to next track characters
			pop	bc
			djnz	pmidmicro_putmapsup	;Loop for all maps
			ret

;******
;pause: Pause the game, and offer the chance to abort
;******
pause:			ld	hl,1900h+77		;VDP address, character 77 in second third
			ld	c,5			;Number of rows
			ld	a,(int_ticks)		;number of int_ticks
			push	af			;Keep it in stack
			ld	a,160			;Starting character for pause message
pause_putblock_outer:	push	bc
			push	hl
			push	af
			ld	a,(VDPportWRITE)	;Get VDP write port
			ld	c,a			;Keep it in C
			ld	a,l			;Low part of address
			di				;Disable interrupts
                	out     (c),a			;Select low part of VRAM address to write to
                	ld	a,h
			or	64
			ei				;Enable interrupts
			out     (c),a			;Select high part of VRAM address to write to
			dec	c			;Data port instead of control
			pop	af
			ld	b,6			;Number of characters per row
pause_putblock_inner:	out     (c),a			;Set character in namestable
			inc	a			;Next character to put
			djnz	pause_putblock_inner	;Put all characters
			pop	hl			;Restore address
			ld	bc,32	
			add	hl,bc			;Point to next line
			pop	bc
			dec	c
			jr	nz,pause_putblock_outer	;Do for all lines
			ld	hl,spritesbuffer+16
pause_cleansprites:	ld	a,(hl)			;Get sprite vertical position
			inc	l
			cp	78
			jr	c,pause_no3		;Above 78: don't clear it
			cp	120
			jr	nc,pause_no3		;Below 120: don't clear it
			ld	a,(hl)			;Get horizontal position
			inc	l
			inc	l
			cp	104			;Left of 104: don't clear it
			jr	c,pause_no1
			cp	152			;Right of 152: don't clear it
			jr	nc,pause_no1
			xor	a
			ld	(hl),a			;Clear sprite while we're in pause
			jr	pause_no1
pause_no3:		inc	l
			inc	l
pause_no1:		inc	l			;Point to next sprite
			jr	nz,pause_cleansprites	;Repeat for all sprites

			ld	hl,1b00h		;VDP memory address for sprites table
			ld	de,spritesbuffer+16
			ld	bc,112
			call	ldrvrm			;Dump sprites to VRAM
pauseback:		call	wait_nokey		;wait until no key is being pressed
			xor	a
			ld	(int_ticks),a		;Clear ticks variable
			ld	bc,mpause_pause
			call	printmessage		;Print pause message
			call	waitkey_60		;Wait for a key, taking into consideration the 60Hz models
			ld	a,(inAA)
			or	2
			out	(#AA),a			;select row 2
			in	a,(#A9)			;read row into A

			and	64			;Check key 'A'
			jr	nz,pause_end		;It wasn't A: go to end the pause
			call	wait_nokey		;wait until no key is being pressed
			xor	a
			ld	(int_ticks),a		;Clear ticks variable
			ld	bc,mpause_abort
			call	printmessage		;Print abort message
pause_abort:		call	waitkey_60		;Wait for a key, taking into consideration the 60Hz models

			ld	a,(inAA)		;Get high bits for key reading
			ld	b,a			;keep in register B
			or	4
			out	(#AA),a			;select row 4
			in	a,(#A9)			;read row into A
			and	8			;Check key 'N'
			jr	z,pauseback		;No: Back to pause
			ld	a,b
			or	5
			out	(#AA),a			;select row 5
			in	a,(#A9)			;read row into A
			ld	c,a			;row 5 in register C
			and	64			;Check key 'Y'
			jr	z,abort_tomenu		;Yes: Abort to Menu
			ld	a,c			;recover row 5
			and	2			;Check key 'T'
			jr	nz,pause_abort		;Not Y,N, or T -> keep waiting
			ld	a,(recording)
			cp	2
			jr	z,notreallyaborting	;Aborting from a Replay doesn't count
			inc	a
			ld	(aborted),a		;Mark that we have aborted (so we don't have the right to see the special picture)
notreallyaborting:	pop	hl			;Trash return address
			pop	af			;Trash int_ticks
			jp	end_stage		;Track: Abort this track
abort_tomenu:		ld	a,(has32k)
			ld	(recording),a		;Disable replay if we're aborting from it
			ld	hl,end_stage_normal
			ld	(end_stage_redirect),hl	;Restore end_stage routine if we were replaying
			pop	hl			;Trash return address
			pop	af			;Trash int_ticks
			call	silencia_ay		;Silence the PSG
			jp	menu			;Go to menu

pause_end:		call	wait_nokey		;wait until no key is being pressed
			ld	hl,spritesbuffer+19	;Address for first sprite color
			ld	a,14			;Gray color for tyres
			ld	b,28			;Number of sprites to restore
pause_restoresprites:	ld	(hl),a			;Restore sprite color
			inc	l
			inc	l
			inc	l
			inc	l			;Point to next sprite
			djnz	pause_restoresprites	;Do for all sprites
			ld	hl,1b00h		;Sprites table
			ld	de,spritesbuffer+16
			ld	bc,112
			call	ldrvrm			;Restore sprites in VDP
			pop	af			;Get int_ticks from stack
			ld	(int_ticks),a		;Restore it
			ret

;***********
;waitkey_60: Wait for a key, reseting the ticks counter to avoid 60Hz music acceleration
;***********
waitkey_60:		halt				;Pause a frame
			ld	a,(int_ticks)		;Get ticks counter
			ld	hl,msxHZ
			cp	(hl)			;Counter needs restart?
			jr	nz,esta_noticks32
			xor	a			;If it's the same
			ld	(int_ticks),a		;Restart counter
esta_noticks32:		call	check_key		;Check if a key has been pressed
			jr	z,waitkey_60		;If not, keep trying
			ret

;********
;get1key: Get a key for redefinition
;********
redef_repeated:		pop	bc
			pop	hl			;Restore registers
redef_poll_pause:	call	micropause		;Do a pause 
			jr	redef_poll		;Start polling again

get1key:		exx				;Use alternate set of registers
redef_poll:		call	get_key			;get key row in A', scan in A, always with at least a "0" bit
			ld	c,8			;Number of bits
redef_bitloop:		rrca				;Get bit
			jr	nc,redef_keyfound	;When bit 0 is found, that key has been pressed
			dec	c			
			jr	redef_bitloop		;Loop all 8 bits

redef_keyfound:						;Key found in A'C
			ex	af,af'			;Key found in AC
			push	af
			push	bc

			and	15			;Remove high bits
			add	a,a
			add	a,a
			add	a,a			;Multiply row by 8
			add	a,c			;Add bit number, so A = key pressed
			ld	(theredefkey),a		;Keep it
			ld	b,a			;B = temporal key
			ld	a,(keysredefined)
			ld	c,a
			inc	c			;C = number of checks +1
			ld	a,b
			ld	b,0
			ld	hl,keyreadbuffer	;Buffer where read keys are stored
redef_checkrepeated:	cpi				;Compare with current key
			jp	po,redef_keyok
			jr	z,redef_repeated	;If the key is found, it means the key has already been used
			jr	redef_checkrepeated	
redef_keyok:		dec	hl
			ld	(hl),a			;Store found key in the table.
			ld	hl,keysredefined
			inc	(hl)			;Increase the number of defined keys
			pop	bc			;C = bit number
			pop	af			;A = row
			ld	(iy+0),a		;Store row (with high bits)
			ld	a,1			;Bit position
redef_adjust:		rrca				;Move bit
			dec	c			
			jr	nz,redef_adjust		;Do it C times
			ld	(iy+1),a		;Store bit mask
			neg
			ld	(iy+2),a		;Store negative bit mask
			ld	a,(theredefkey)		;Get keycode
			add	a,(keytable%256)-1	;Get character for the keycode
			ld	l,a
			ld	h,keytable/256
			ld	a,(hl)			;Character in A
			ld	bc,3
			add	iy,bc			;Point to next key
			exx				;Restore original register set
			ret

;***************
;keyclick_sound: sound effect beep
;***************
keyclick_sound:		ld	e,0
			ld	a,(snd_state)		;Get sound state
			ld	d,a
			and	2			;Are we in collision?
			jr	z,snd_speed2
			ld	e,20			;Additional value for collision
snd_speed2:		ld	a,(snd_speed)		;Sound depends on current speed
			add	a,e			
			ld	e,a
			jr	nz,youwillbeep
			ld	a,d
			and	4			
			ret	nz			;If state has bit 2 set, don't beep
youwillbeep:		ld	a,36
			sub	e
			add	a,a
;			add	a,a
			ld	e,a
			add	a,a
			ld	d,a
			ld	c,1
			call	Beep

PlayOut:		ld	a,(inAA)
			and	127
			out	(0AAh),a		;Clear keyclick bit
			ret

Beep:			ld	a,(inAA)
			or	128
			out	(0AAh),a		;Keyclick bit at 1, border remains 0
			ld	b,e
Beep0:			;ld	a,r
			nop
			nop
			djnz	Beep0
			jr	Beep1

Beep1:			ld	a,(inAA)
			and	127
			out	(0AAh),a		;Keyclick bit at 0, border remains 0
			ld	b,d
Beep2:			;ld	a,r
			nop
;			nop
			nop
			nop
			djnz	Beep2
			dec	c
			ret	z
			jr	Beep

;**********
;check_key: check if there's any key pressed. Returns Z if no key is pressed, NZ if a key is pressed.
;**********
check_key:		push	bc
			ld	c,255		;All bits 1
			ld	b,12		;Check 12 key rows
			ld	a,(inAA)	;Get high bits, row 0
checkkeyrow:		out	(#AA),a		;Select row
			ex	af,af'
			in	a,(#A9)		;read row into A
			and	c		;And value with previous C
			ld	c,a		;Update C
			ex	af,af'
			inc	a		;Next row
			djnz	checkkeyrow	;Loop for all rows
			ld	a,c		;Result in A
			pop	bc
			inc	a		;Z = no key pressed, NZ = key pressed
			ret

;********
;get_key: Wait for a key to be pressed, get it's scan and row
;********
get_key:		ld	b,12		;Check 12 key rows
			ld	a,(inAA)	;Take high bits of AA port, select row 0
checkkeyrow2:		out	(#AA),a
			ex	af,af'
			in	a,(#A9)		;read row into A
			cp	255
			ret	nz		;A = SCAN, A' = ROW
			and	c
			ld	c,a
			ex	af,af'
			inc	a		;Next row
			djnz	checkkeyrow2	;Loop all rows
			jr	get_key		;Try again

;***********
;wait_nokey: wait until no key is being pressed
;***********
wait_nokey:		push	af		;Save AF,AF' and HL
			ex	af,af'
			push	af
			push	hl
wait_nokey2:		halt			;Wait for an interrupt
			ld	a,(int_ticks)	;Get ticks
			ld	hl,msxHZ
			cp	(hl)		;Value = hertzs per frame?
			jr	nz,wait_noticks0;No: Do nothing
			xor	a		;Yes: clear it
			ld	(int_ticks),a
wait_noticks0:		call	check_key	;Check if there's any key pressed
			jr	nz,wait_nokey2	;Loop until a key has been pressed
			pop	hl		;Restore registers
			pop	af
			ex	af,af'
			pop	af
			ret

;*********
;wait_key: Wait until a key is pressed
;*********
wait_key:		call	check_key	;Check if a key has been pressed
			jr	z,wait_key	;Keep checking if not
			ret

;***********
;micropause: Do a 10 frames (0.2 seconds in 50Hz) pause
;***********
micropause:		push	bc
			ld	b,10			;Number of frames
mpause:			halt				;Wait a frame
			djnz	mpause			;Loop 10 times
			pop	bc
			ret

;**********
;checkgirl: Check if we must the special image, and show it if the answer is yes.
;**********

checkgirl:		ld	a,(aborted)		;Check if any track has been aborted
			and	a
			ret	nz			;No aborted track allowed
			ld	ix,car1			;First car
			ld	de,car2-car1
			ld	bc,4*256		;C=0, B=4 (number of cars)
p3_getwinner:		ld	a,(ix+SP_CONTOPT)
			cp	8
			jr	nz,p3_notcpuhard
			inc	c
p3_notcpuhard:		ld	a,(ix+SP_MAPPOINTS)	;position > 0 = winner
			and	a
			jr	nz,p3_getwinnerno1
			bit	7,(ix+SP_STATE)
			ret	z			;non-human winner = return
			ld	a,(ix+SP_POINTS)
			ld	(p3_points1),a		;get first place points
			jr	p3_getwinnerno2
p3_getwinnerno1: 	dec	a
			jr	nz,p3_getwinnerno2
			ld	a,(ix+SP_POINTS)
			ld	(p3_points2),a		;get second place points
p3_getwinnerno2: 	add	ix,de			;Point to next car
			djnz	p3_getwinner		;Loop for all cars
			dec	c
			ret	nz			;exit if number of CPU hard != 1
			ld	a,(mode)
			sub	1
			jr	c,p3_checkf1		;If mode was 0, check F1 conditions
			ret	nz
						;check GP2
			ld	a,(championship)
			cp	1
			jr	z,xend			;Condition 1: GP2, HARD championship, winner
			ret
p3_checkf1:		ld	a,(championship)
			and	a
			ret	z			;no glory for EASY championship
			cp	3
			ret	z			;no glory for PERSONAL championship either
			dec	a
			jr	z,p3_getpoints1
							;check F1 hard
			ld	a,(p3_points2)
			ld	b,a
			ld	a,(p3_points1)
			sub	b
			cp	4
			jr	c,xend			;Condition 2: F1, HARD championship, winner by at least 5 points over the second
			ret
p3_getpoints1:		ld	a,(p3_points1)		;Condition 3: F1, COMPLETE championship, win all 72 points (8 victories with 8 top laps)
			cp	72
			ret	nz			;Girl 3 (interlaced): only with 8*(6+3)=72 points
			;Check different end conditions
xend:			call	vdpdis			;Disable VDP
			ld	hl,girlch01
			call	deexo_buffer		;Uncompress first part of girl's characters (1st and 2nd third)
			ld	hl,0			;VRAM address
			ld	bc,4096			;Number of bytes
			call	ldrvrmbuffer		;Write first part of girl's characters in VRAM
			ld	hl,girlch2
			call	deexo_buffer		;Uncompress second part of girl's characters (3rd third)
			ld	hl,1000h		;VRAM address
			call	ldrvrmbuffer2k		;Write second part of girl's characters in VRAM
			ld	hl,girlco01
			call	deexo_buffer		;Uncompress first part of girl's colors (1st and 2nd third)
			ld	hl,2000h		;VRAM address
			ld	bc,4096			;Number of bytes
			call	ldrvrmbuffer		;Write first part of girl's colors in VRAM
			ld	hl,girlco2
			call	deexo_buffer		;Uncompress second part of girl's colors (3rd third)
			ld	hl,3000h		;VRAM address
			call	ldrvrmbuffer2k		;Write second part of girl's colors in VRAM
			ld	bc,3			;C=3 thirds, B = 0 = 256 characters
			ld	hl,screenbuffer
p3_fillscreen:		ld	(hl),l			;Fill screenbuffer with 0,1,2,3...
			inc	hl
			djnz	p3_fillscreen		;Loop all characters in third
			dec	c
			jr	nz,p3_fillscreen	;Loop all thirds
			ld	hl,1800h		;VRAM address
			ld	de,screenbuffer
			ld	bc,768			;Number of bytes
			call	ldrvrm			;Write names to VRAM
			call	vdpena			;Enable VDP
			ld	b,150
pausababe:		halt
			djnz	pausababe		;Pause at least 150 frames (3 seconds in 50Hz)
xwait_key:		call	check_key		
			jr	z,xwait_key		;Wait for a key to be pressed
     			ret
free0:

;********
;INSROM1: The second 16k block starts here, this is the only visible block at the beginning
;********

			org	16384			;ROM init at 4000h
			defb	"AB"			;MSX ROM identifier
			defw	start			;Code start point
			defw	0
			defw	0
			defw	0
			defw	0
			defw	0
			defw	0
			defb	"I_NEED_SPEED"		;ROM name
			defb	0x1A			;end of ROM name string
start:	        	di
			im	1
			ld      sp,0xf380		; Set stack

   			call	RSLREG      		;read primary slot #
   			rrca     			;move it to bit 0,1 of [Acc]
   			rrca
			and   	3
   			ld   	c,a
   			ld	b,0
   			ld   	hl,EXPTBL   		;see if this slot is expanded or not
   			add   	hl,bc
   			ld   	a,(hl)      		;See if the slot is expanded or not
   			and   	$80
   			or   	c      			;set MSB if so
   			ld   	c,a     		;save it to [C]
   			ld	a,l
   			add	a,4
   			ld	l,a			;Point to SLTTBL entry
   			ld   	a,(hl)      		;Get what is currently output to expansion slot register
   			and   	12
   			or   	c      			;Finally form slot address
;   			ld 	(_slot2address), a	; save it!
   			ld   	h,80H
   			call   	ENASLT      		;enable page 2
							; Disable keyboard click
			xor     a
        		ld      (CLIKSW),a		; Disable keyboard click - system variable
	        	call    GICINI			; Init PSG registers
	        	xor     a
	        	ld      r,a			; Init memory refreshment register R - [0-127]
	        					; Hide function keys
	        	call    ERAFNK			; BASIC: KEY OFF
			call	INIGRP			; SCREEN 2

			ld 	a, ($002b)
			add	a,128			; The highest bit is 1 for 50Hz, 0 for 60Hz
			ld	a,6
			sbc	a,0
			ld 	(msxHZ), a		; 5 for 50Hz, 6 for 60Hz
			ld	hl,(6)			;Get VDP ports bases from BIOS
			inc	h
			inc	l			;Increase them to get control ports instead of data port
			ld	(VDPportREAD),hl	;Put them in RAM

                        call    search_slot             ;Look for our PAGE 0 slot
                        call    setrompage0             ;Bye Bye BIOS
                        di				;Disable interrupts
			ld	hl,int_end
			ld	(interrupt_redir),hl	;No extra routine after interrupt

							;Right here, we have:
							; INSROM0 in page 0
							; INSROM1 in page 1
							; INSROM2 in page 2
							; RAM	  in page 3, EXPTBL still exists

			; check if we have at least 16K of RAM
			
			ld	hl,check16kRAM		;Address to check RAM
			ld	a,(hl)			;Get content
			xor	170			;Change it
			ld	(hl),a			;Store new content
			cp	(hl)			;Compare register with what we've written
			jp	z,atleast16k		;If it's the same, we have at least 16k

							;Here we only have 8K
			xor	a
			ld	hl,0E000h		;Buffer in available RAM
			ld	de,0E001h		;Buffer+1
			ld	bc,2047			;Size -1
			ld	(hl),a			;Set 0
			ldir				;Fill all buffer with zeros
			ld	hl,0
			ld	bc,2048
			call	ldrvrm8k		;Clear first third graphics
			ld	hl,800h
			ld	bc,2048
			call	ldrvrm8k		;Clear second third graphics
			ld	hl,1000h
			ld	bc,2048
			call	ldrvrm8k		;Clear third third graphics
			ld	hl,2000h
			ld	bc,2048
			call	ldrvrm8k		;Clear first third attributes
			ld	hl,2800h
			ld	bc,2048
			call	ldrvrm8k		;Clear second third attributes
			ld	hl,3000h
			ld	bc,2048
			call	ldrvrm8k		;Clear third third attributes
			ld	hl,1b00h
			ld	bc,128
			call	ldrvrm8k		;Clear sprites
			call	pmess_select_center	;Centered text
			ld	bc,mneed_ram
			call	printmessage		;Print I NEED RAM message
			di			
			halt				;Show message and hang
			
			; Now we want to look for extra RAM in page 2, without using BIOS
atleast16k:		ld	hl,EXPTBL	
			in	a,(#A8)
			ld	(a8romslot),a		;Store initial ROM slot
			rlca
			rlca				;Get RAM slot for page 3 in bits 1 and 0
			and	3
			add	a,l
			ld	l,a
			ld	a,(a8romslot)		;Get initial ROM slot
			ld	e,a			;Keep it in register E
			bit	7,(hl)
			jr	z,noexpanded
			rlca
			rlca
			and	192			;Get INSROM2 slot
			ld	d,a			;Into bits 7 and 6
			ld	a,(a8romslot)
			and	63			;Clear bits 7 and 6
			or	d			;Enter INSROM2 slot
noexpanded:		ld	(a8rompreslot),a	;Store preslot

			out	(#A8),a			;Set page 2 slot into page 3 too
			ld	a,(65535)
			cpl				;Get INSROM2 subslot
			
			ld	d,a			;keep it in D (variables are out of page now!)
			ld	a,e			;restore slot
			out	(#A8),a			;restore RAM in page 3
			ld	a,d			;get subslot in A again
			and	48			;Keep only page 2 bits
			ld	(a8romsubslot),a	;Got it!

				;Now start looking for RAM

			ld	ix,EXPTBL
			;in	a,(#A8)
			ld	a,e			;Get slots again
			and	207			;Clear bits 5 and 4, to start with slot 0
			ld	b,4
searchslotloop:		ld	c,a
			push	bc
							;C now has the current slot in bits 54, and the rest of slots for
							;INSROM0 in page 0, INSROM1 in page 1, RAM in page 3
			bit	7,(ix)			;Check if there are subslots
			jr	nz,nosimplecheck
							;There are no subslots
			out	(#A8),a

			ld	hl,32768		;Check address 1
			ld	a,(hl)			;Get content
			xor	255
			ld	(hl),a			;Store inverted content
			cp	(hl)			;Check if value has changed
			jr	nz,tonoextraram2	;NO: there's no RAM here
			xor	255
			ld	(hl),a			;Restore if it was RAM
			ld	hl,40001		;Check address 2
			ld	a,(hl)			;Get content
			xor	170
			ld	(hl),a			;Store content with some bits changed
			cp	(hl)			;Check if value has changed
			jr	nz,tonoextraram2	;NO: there's no RAM here
			xor	170
			ld	(hl),a			;Restore if it was RAM
			inc	h			;Check address 3
			ld	a,(hl)			;Get content
			xor	85
			ld	(hl),a			;Store content with some bits changed
			cp	(hl)			;Check if value has changed
tonoextraram2:		jp	nz,noextraram2		;NO: there's no RAM here
			xor	85
			ld	(hl),a			;Restore if it was RAM
	
			ld	d,c			;slot
			ld	e,c			;preslot = slot
			ld	c,0			;noexistant subslot

			ld	a,(a8rompreslot)
			cp 	e			;Check if slot is the same that has INSROM2
			jr	z,noextraram2		;Yes: Then we're executing the game from RAM, and no RAM has been found here.
			push	bc			;Fake BC in stack
			jr	diffpreslot		;Ram Found > go to store it!

nosimplecheck:		ld	a,c
			ld	d,a			;Keep RAMSLOT in D
			rlca
			rlca
			and	192			;Isolate slot bits for page 2 in position of page 3
			ld	b,a			;Keep it in B
			ld	a,c
			and	63
			or	b			;Copy page 2 slot to page 3
			out	(#A8),a			;Address it
			ld	e,a			;Keep RAMPRESLOT in E
			ld	a,(65535)
			cpl				;Get subslots in the testing slot
			and	207			;Clear bits 54
			ld	c,a			;Keep in C
			ld	b,4			;Number of subslots to test
searchsubslotloop:	ld	(65535),a		;Set Subslot
			ld	a,d			;Get RAMSLOT
			out	(#A8),a			;Restore RAM in page 3
			push	bc			;Keep BC

			ld	hl,32768		;Check address 1
			ld	a,(hl)			;Get content
			xor	255
			ld	(hl),a			;Store inverted content
			cp	(hl)			;Check if value has changed
			jr	nz,noextraram		;NO: there's no RAM here
			xor	255
			ld	(hl),a			;Restore if it was RAM
			ld	hl,40001		;Check address 2
			ld	a,(hl)			;Get content
			xor	170
			ld	(hl),a			;Store content with some bits changed
			cp	(hl)			;Check if value has changed
			jr	nz,noextraram		;NO: there's no RAM here
			xor	170
			ld	(hl),a			;Restore if it was RAM
			inc	h			;Check address 3
			ld	a,(hl)			;Get content
			xor	85
			ld	(hl),a			;Store content with some bits changed
			cp	(hl)			;Check if value has changed
			jr	nz,noextraram		;NO: there's no RAM here
			xor	85
			ld	(hl),a			;Restore if it was RAM

			ld	a,c
			and	48			;Keep only page 2 bits
			ld	c,a

			ld	a,(a8rompreslot)
			cp 	e			;Compare if preslot is different from INSROM2's one
			jr	nz,diffpreslot		;Yes: We've found RAM, go take it!
			ld	a,(a8romsubslot)
			cp	c			;Compare if sublot is different from INSROM2's one
			jr	z,noextraram		;No: slot and subslot are the same, so we haven't really found RAM

diffpreslot:		ld	a,e
			ld	(a8rampreslot),a	;Store RAM PRESLOT
			ld	a,d
			ld	(a8ramslot),a		;Store RAM SLOT
			ld	a,c
			ld	(a8ramsubslot),a	;Store RAM SUBSLOT
			pop	bc
			pop	bc
			ld	a,1			;Signal extra RAM found
			jr	extraramcont		;End RAM search
			
noextraram:		pop	bc			;Get current sublot
			ld	a,c
			add	a,16			;Next page 2 subslot
			ld	c,a
			ld	a,e
			out	(#A8),a			;Address preslot
			ld	a,c
			djnz	searchsubslotloop	;Keep searching the subslot
			ld	a,d
			out	(#A8),a			;Restore RAM in page 3
noextraram2:		pop	bc			;Get current slot
			inc	ix			;Point to next EXPTBL value
			ld	a,c
			add	a,16			;Next slot
			dec	b
			jp	nz,searchslotloop	;Loop for all slots
			xor	a			;Signal no extra RAM found

extraramcont:		ld	(has32k),a		;1 if it has 32K of RAM, 0 if it hasn't
			ld	(recording),a		;record if we have memory

			rst	8			;Set ROM in page 2
			di

                        ld	sp,0fef0h		;put stack down
                        ld      a,(BIOSVDPREG+1)        ;Take this value
                        ld      (VDPREG),a              ;Store at new location

			ld	a,(VDPportWRITE)	;Get VDP write port
			ld	c,a
			xor	a			;Black border
                	out     (c),a
			ld	a,128+7			;Mode Register 7
			nop
			nop
                	out     (c),a

			ld 	a,(msxHZ)		;Preserve msxHZ value
                	push	af
			ld	hl,varpacked
			ld	de,variables
			call	deexo			;Uncompress initial variable state
			pop	af
			ld 	(msxHZ), a		;Restore msxHZ value

			in	a,(#AA)			;Read AA port 
			and	#F0			;mask high bits
			ld	(inAA),a		;Keep them for future access
                        ld      c,a
                        ld      b,16			;number of keys to be fixed
                        ld      hl,keys1
fixinitialkeys:         ld      a,(hl)			;Get initial value
                        or      c			;Add high bits to port
                        ld      (hl),a			;Store fixed value
                        inc     hl
                        inc     hl
                        inc     hl			;Point to next key
                        djnz    fixinitialkeys		;Repeat for all keys

			ld	hl,end_stage_normal
			ld	(end_stage_redirect),hl	;stage_normal initial redirection
                	ld	hl,int_end
                	ld	(interrupt_redir),hl	;interrupt initial redirection

			ld	hl,spriteformspacked
			call	deexo_buffer		;Uncompress sprite forms to buffer
			ld	hl,03800h		;VDP address
			call	ldrvrmbuffer2k		;Write sprite forms to VRAM
			call	pmess_select_center	;Centered text
			call 	vdpdis			;Disable VDP

			call	cls			;Clear screen
			ld	hl,logo
			ld	de,mapa
			call	deexo			;Uncompress CEZGS logo
			
			ld	bc,38*8			;Number of bytes
			ld	de,mapa
			ld	hl,808h			;VDP address, second third characters
			call	ldrvrm			;Move data to VRAM
			ld	bc,38*8			;Number of bytes
			ld	de,mapa+38*8
			ld	hl,2808h		;VDP address, second third colors
			call	ldrvrm			;Move data to VRAM
			ld	hl,1800h		;VDP address, names
			ld	de,mapa+38*16
			ld	bc,768			;All 768 bytes
			call	ldrvrm			;Move data to VRAM
			call 	vdpena			;Enable VDP, to show CEZGS logo
			ei				;Enable interrupts
			ld	a,(msxHZ)		;Get msxHZ
			ld	b,30			;3 seconds
logo_wait3seconds:	halt
			halt
			halt
			halt
			halt				;5 halts
			cp	6
			jr	nz,logo_no60
			halt				;Do an extra halt for 60Hz
logo_no60:		djnz	logo_wait3seconds	;Wait 3 seconds

;*****
;menu: Main menu routine
;*****
menu:			di				;Disable interrupts
			call	cls			;Clear screen
			ld	hl,song_menu
			ld	de,DEC_PT3
			call	deexo			;Uncompress menu song
			ld	hl,DEC_PT3-100		; HL <- initial address of module-100
			xor	a
			ld	(PT3_SETUP),a		;Loop please
			ld	(int_ticks),a		;Don't start accelerated
			call	PT3_INIT		; Init PT3 player
			ld	hl,int_menu
                	ld	(interrupt_redir),hl	;Set menu interrupt routine
			ei				;Enable interrupts: Play the music!
			ld	hl,menuchars
			call	deexo_buffer		;Uncompress menu logo and copyright chars
			ld	hl,8			;VRAM address
			ld	bc,448
			call	ldrvrmbuffer		;Logo chars, 1st third
			ld	hl,1000h+8		;VRAM address
			ld	bc,256
			ld	de,buffer+448
			call	ldrvrm			;Copyright chars, 3rd third
menuprint2:		ld	hl,screenmenu
			ld	de,screenbuffer
			call	deexo			;Uncompress menu screen
			ld	hl,1800h
			ld	bc,768
			ld	de,screenbuffer
			call	ldrvrm			;Menu screen ready...
			ld	bc,mmenu_fixed
			call	printmessage_nr		;Print menu message
menuprint:		call	menuprintcontrols	;Print menu controls
menu_resetc:		ld	c,0
menu_outwaitres:	ld	b,4
menu_inwaitres: 	push	bc
			ld	bc,mmenu_0start
			call	printmessage_nr		;Print "0 TO START" message
			ld	a,(inAA)
			out	(#AA),a			;select row 0
			in	a,(#A9)			;read row into A, keys: 76543210
			rrca
			jp	nc,start_game		;0 pressed: start game
			ld	b,4			;Number of cars
			ld	ix,car1			;First car
			ld	de,car2-car1
menu_testkey:		rrca				;Check if we're changing the control of a ar
			jp	nc,menu_changecontrol	;Yes: Go to change its control
			add	ix,de			;No: Point to next car
			djnz	menu_testkey		;Try all cars
			rrca				;Check for key 5
			jp	c,menu_noredefine	;No: Don't redefine
			
			call	pmess_select_center	;Centered text
			call	wait_nokey		;Wait for no key to be pressed
			ld	bc,mmenu_keyfor
			call	printmessage_nr		;Print the "KEY FOR " message
			xor	a
			ld	(keysredefined),a	;Reset the number of defined keys so far
			ld	ix,car1			;First car
			ld	b,4			;Number of cars
			
menured_checkkeys:	ld	a,(ix+SP_CONTOPT)	;For each car, take the control option
			ld	hl,mmenu_keys1_def	;Message with current KEY1 keys
			ld	iy,keys1		;Key data to be patched
			and	a			;0?
			call	z,redefinekeys		;Yes (keys1): go to redefine
			ld	hl,mmenu_keys2_def	;Same with the three key control routines
			ld	iy,keys2
			dec	a
			call	z,redefinekeys
			ld	hl,mmenu_keys3_def
			ld	iy,keys3
			dec	a
			call	z,redefinekeys
			ld	de,car2-car1
			add	ix,de			;Point to next car
			djnz	menured_checkkeys	;Repeat for all cars

			ld	hl,mmenu_keypause	;Message with current control keys
			ld	iy,keys4		;Key data to be patched
			call	redefineextra		;Redefine the extra keys
			jp	menu_changeoption2	;Go back to menu, with a change

;**************
;redefineextra: Redefine the extra keys
;**************
redefineextra:		push	af
			push	bc
			push	ix
			ld	bc,mmenu_rede_pause	;Messages for extra keys
			jp	rede_nextkey

;*************
;redefinekeys: Redefine 4 movements keys
;*************
redefinekeys:		push	af
			push	bc
			push	ix
			ld	bc,mmenu_rede_right	;Messages for movement keys
rede_nextkey:		push	hl
			push	bc
			push	hl
			ld	a,CHR_INTERR
			ld	(hl),a			;Put an interrogation symbol in questioned key
			push	bc
			push	hl
			call	menuprintcontrols	;Print menucontrols to display that symbol
			call	micropause		;Make a small pause 
			pop	hl
			pop	bc
			ld	hl,800h+59*8		;Fixed message position, right after "KEY FOR "
			ld	(pmess_address),hl	;Store it
			call	printmessage_nr		;Print the name of the key we're looking for
			call	get1key			;Get 1 key that's not been pressed before
			pop	hl		
			ld	(hl),a			;Put the new key in the place of the interrogation symbol
			call	menuprintcontrols	;Print the menucontrols with the new key
			call	micropause		;Make a small pause 
			pop	bc
			ld	hl,7
			add	hl,bc
			ld	b,h
			ld	c,l			;Advance to next movement/special key name
			pop	hl			;HL points to current key
			inc	hl
			ld	a,c
			cp	mmenu_rede_pause%256	;Check if we're at the last movement key
			jr	z,rede_endrede		;Yes: end redefinition
			cp	mpause_pause%256	;Check if we're at the last extra key
			jr	nz,rede_nextkey		;If not, keep redefining
rede_endrede:		pop	ix
			pop	bc
			pop	af
			ret

;*******************
;menu_changecontrol: Change control of 1 of the 4 players, to the next legal value
;*******************
menu_changecontrol:	ld	a,(ix+SP_CONTOPT)		;Current control
			inc	a				;Increase it
			cp	5				;Kempston?
			jr	nz,menu_noinca
			inc	a				;Kempston doesn't exist on MSX!
menu_noinca:		cp	10				;Reached 10?
			jr	nz,menu_noxora
			xor	a				;Yes: set to 0		
menu_noxora:		ld	(ix+SP_CONTOPT),a		;Store new value
			and	a
			call	menu_testlegal			;Check if the new value is legal
			jr	c,menu_changecontrol		;If not, change again
			jr	menu_changeoption2		;Legal: go back to menu, with a change

;****
;menu continues here...
menu_noredefine:	rrca					;6 pressed?
			jr	c,menu_nochangecat
								;Yes: change category
			ld	a,(mode)			;Get current one
			dec	a				;Decrease value
			and	3				;Keep it in 0-3 range
			ld	(mode),a			;Store new value
			jr	menu_changeoption2		;Go back to menu, with a change

menu_nochangecat:	rrca					;7 pressed?
			jr	c,menu_nochangechamp
								;Yes: change championship
			ld	a,(championship)		;Get current one
			inc	a				;Increase value
			and	3				;Keep it in 0-3 range
			ld	(championship),a		;Store new value
			jr	menu_changeoption2		;Go back to menu, with a change

menu_changeoption:	;ldir
menu_changeoption2:	call	micropause			;Do a small pause
			call	menuprintcontrols		;Print menu controls
			pop	bc				;Restore counter
			ld	a,c
			and	3
			ld	c,a				;Keep only the lower 2 bits of C
			push	bc
menu_nochangechamp:	ld	a,(inAA)
			inc	a
			out	(#AA),a				;select row 1
			in	a,(#A9)				;read row into A, keys: ;][\=-98
			rrca					;8 pressed?
			jp	nc,menu_personalize		;Yes: Go to personize routine
			pop	bc
					;MENU KEYS ACTION END
			halt
			halt
			halt					;Wait 3 frames
			dec	b				;Time counter
			jp	nz,menu_inwaitres		;Loop
			inc	c				;Another time counter
			ld	a,c
			cp	50
			jp	nc,menu2			;After some time passes, go to menu2 (credits)
			and	3				;Get two lower bits from the counter
			rrca
			jr	c,menu_centerit			;If bit is 1, center chars for "0 TO START"
			jr	z,menu_leftit			;If value is 0, left inclination chars for "0 TO START"
			call	pmess_select_right		;Otherwise, right inclination chars for "0 TO START"
			jp	menu_outwaitres
menu_centerit:		call	pmess_select_center
			jp	menu_outwaitres
menu_leftit:		call	pmess_select_left
			jp	menu_outwaitres

;*****************
;menu_personalize: Personalize the personal championship
;*****************
menu_personalize:	ld	hl,mapssmall
			ld	de,mapa
			call	deexo				;Uncompress micromaps into mapa buffer
			ld	hl,1400h
			ld	de,mapa
			ld	bc,1024
			call	ldrvrm				;128-255: minimaps chars in 3rd third, and in mapa to create the needed ones in 1st and 2nd third

			ld	hl,screenpersonal
			ld	de,screenbuffer+96
			call	deexo				;Uncompress personalization screen
			ld	hl,screenbuffer+96+544		;Lower micromaps position in screen (3rd third)
			ld	a,128				;First character
			ld	c,4				;4 lines
putmicromapsdown_outer:	ld	b,32
putmicromapsdown_inner:	ld	(hl),a				;Set character
			add	a,4				;Point to next character
			inc	hl
			djnz	putmicromapsdown_inner		;Do it for 32 characters per line
			sub	127				;Point to next row of characters
			dec	c
			jr	nz,putmicromapsdown_outer 	;Loop for all lines
			
			ld	hl,screenbuffer+192		;Middle micromaps position in screen (1st third, 6th line)
			ld	a,128				;First character
			ld	c,4				;4 lines
putmicromapsdown_outer2:ld	b,32
putmicromapsdown_inner2:ld	(hl),a				;Set character
			add	a,4				;Point to next character
			inc	hl
			djnz	putmicromapsdown_inner2		;Do it for 32 characters per line
			sub	127				;Point to next row of characters
			dec	c
			jr	nz,putmicromapsdown_outer2 	;Loop for all lines

			ld	hl,1860h			;Don't touch the first three lines (logo+blank line from menu)
			ld	bc,672
			ld	de,screenbuffer+96
			call	ldrvrm				;Personalize screen ready...

			call	pmess_select_center		;Centered text
			ld	bc,mpers_init
			call	printmessage_nr			;Print personalization message
			ld	a,3
			ld	(championship),a		;Select PERSONAL championship

mpers_tracks_add:	call	putmiddlemicromaps		;Put micromaps of the current personal championship in the middle of the screen
			call	updatepersdata			;Update data
mpers_tracks:		ld	de,championship_personal+14	
			ld	a,(person_maps)			;Get number of maps
			cp	8
			jr	nz,mpers_nodec
			dec	a			;If it's 8, decrease it
mpers_nodec:		add	a,e
			ld	e,a			;Point to magenta map
mpers_trackget:		call	getkey_personalize	;B=KEY 0-8 or 9 for BACKSPACE, carry back if ENTER
			call	micropause
			jp	c,mpers_laps		;Enter -> Laps
			ld	a,b
			and	a
			jr	z,mpers_trackget	;0 not used
			cp	9
			jr	z,mpers_erasetrack	;9->erase
			dec	a
			add	a,a			;2*(number-1)
			ld	(de),a			;Store map code
			ld	a,-8
			add	a,e
			ld	e,a
			ld	a,2			;Default 2 laps for the new map
			ld	(de),a
			jr	mpers_tracks_add

mpers_erasetrack:	ld	a,(person_maps)			;Get number of maps
			and	a
			jr	z,mpers_trackget		;Don't erase if we're at the first map
			dec	a				;Decrease the number of maps
			jr	z,mpers_erasetrackfirst		;First map is a special case
			ld	e,(championship_personal+6)%256	
			add	a,e
			ld	e,a
			xor	a
			ld	(de),a				;0 the map erased
			jr	mpers_tracks_add
mpers_erasetrackfirst:	ld	e,128				;First map characters
			ld	a,209				;Magenta
			call	attrblock4x4			;Put magenta 4x4 block
			ld	e,144				;Second map
			ld	a,17				;Black
			call	attrblock4x4			;Black over the second map
			ld	de,championship_personal+14
			jr	mpers_trackget			;Keep getting tracks
		;Laps
mpers_laps:		ld	e,128				;First character
			ld	hl,championship_personal+6
			ld	a,(person_maps)			;Get number of personal maps
			and	a
			jr	z,mpers_nodec2
			dec	a				;Decrease to start by 0
mpers_nodec2:		add	a,a
			add	a,a
			add	a,a
			add	a,a				;Multiply by 16 (characters for each map)
			add	a,e
			ld	e,a				;Add initial character
			ld	a,113				;Cyan
			push	de
			call	attrblock4x4			;Paint cyan
			pop	de
			ld	a,16
			add	a,e
			ld	e,a				;Update character
;			cp	240
			jr	z,mpers_lapsstart		;If we're at 0, it was the last map, so go to laps
			xor	a
			call	attrblock4x4			;Clear next map
mpers_lapsstart:	ld	hl,championship_personal+6
			ld	de,mpers_laps0			;Point to laps per track message
mpers_laploop:		ld	a,CHR_INTERR
			ld	(de),a			;Put an interrogation over the number of laps we're asking for
			push	bc
			push	de
			push	hl
			ld	bc,mpers_param
			call	printmessage_nr		;Update message to make the interrogation appear
			call	getkey_personalize	;Get a key
			call	micropause		;Do a pause
			ld	a,b
			pop	hl
			pop	de
			pop	bc
			jr	c,mpers_scores_fromenter ;Enter -> Scores
			and	a
			jr	z,mpers_laploop		;No 0
			cp	9
			jr	z,mpers_gobacklap	;Go back
			cp	6
			jr	nc,mpers_laploop	;No 6-8
			ld	(hl),a			;Store number of laps
			add	a,CHR_0			;Turn into a character
			ld	(de),a			;Store it in message, in place of the interrogation
			inc	hl			;Point to the number of laps in next track (in championship data)
			inc	de			;Point to the number of laps in next track (in message)
			ld	a,(hl)			;Get number of laps of the next track
			and	a			
			jr	nz,mpers_laploop	;If it's not zero, ask for it again
			jr	mpers_scores		;Otherwise, go to scores part
mpers_gobacklap:	ld	a,(hl)
			add	a,CHR_0
			ld	(de),a			;Restore message with previous number of laps instead of the interrogation mark
			dec	hl			;Point to the number of laps in previous track (in championship data)
			dec	de			;Point to the number of laps in previous track (in message)
			ld	a,l
			cp	(championship_personal+5)%256	;Check if we were in the first track
			jp	z,mpers_tracks_add	;If so, go back to tracks part
			jr	mpers_laploop		;Go ask for the number of laps again
		;Scores
mpers_scores_fromenter:	ld	a,(hl)
			add	a,CHR_0			
			ld	(de),a			;Turn the last interrogation mark into the number of laps
mpers_scores:		ld	hl,championship_personal+1
			ld	de,mpers_points1	;Point to points per position message
mpers_scoresloop:	ld	a,CHR_INTERR		;Interrogation
			ld	(de),a
			push	bc
			push	de
			push	hl
			ld	bc,mpers_param
			call	printmessage_nr		;Update message to make the interrogation appear
			call	getkey_personalize	;Get a key
			call	micropause		;Do a pause
			ld	a,b
			pop	hl
			pop	de
			pop	bc
			jp	c,menu			;ENTER: We're done
			cp	9
			jr	z,mpers_gobackscore	;Go back
			cp	7
			jr	nc,mpers_scoresloop	;No 7-8
			ld	(hl),a			;Store score
			add	a,CHR_0			;Turn it into a character
			ld	(de),a			;Store it in message, in place of the interrogation
			inc	hl			;Point to the score for the next position (in championship data)
			inc	de			;Point to the score for the next position (in message)
			ld	a,e
			cp	(mpers_points3+1)%256	;Check if we've arrived at third change in scores
			jr	nz,mpers_scores_nochange1
			inc	de
			inc	de			;Change third in message
mpers_scores_nochange1:	ld	a,l
			cp	championship_personal+6
			jr	nz,mpers_scoresloop

mpers_scoresdone:	push	bc
			push	de
			push	hl
			ld	bc,mpers_param
			call	printmessage_nr		;Clear last interrogation
			call	getkey_personalize	;Get a key
			call	micropause
			ld	a,b
			pop	hl
			pop	de
			pop	bc
			jp	c,menu			;ENTER: We're done
			cp	9
			jr	nz,mpers_scoresdone	;Not BACKSPACE key: keep trying
			jr	mpers_gobacknorest	;BACKSPACE: Go back to the score for best lap
mpers_gobackscore:	ld	a,(hl)
			add	a,CHR_0
			ld	(de),a			;Restore previous interrogation
mpers_gobacknorest:	dec	hl			;Point to the score for the previous position (in championship data)
			dec	de			;Point to the score for the previous position (in message)
			ld	a,e
			cp	(mpers_points3+2)%256	;Check if we're changing third
			jr	nz,mpers_scores_nochange2
			dec	de
			dec	de			;Change third in message
mpers_scores_nochange2:	ld	a,l
			cp	(championship_personal)%256
			jr	nz,mpers_scoresloop	;If we were not in the first points, go back to the previous one
			
			ld	a,(person_maps)		;Get number of tracks
			ld	de,mpers_laps0-1	;number of laps message
mpers_gobacklastlap:	inc	de			;Advance a position in message
			dec	a
			jr	nz,mpers_gobacklastlap	;Advance to the last map
			ld	hl,championship_personal+5
			ld	a,(person_maps)
			add	a,l
			ld	l,a			;HL points to the number of laps for the last track
			jp	mpers_laploop		;Go to laps part

;***************
;updatepersdata: Update personalized championship data
;***************
updatepersdata:		ld	de,mpers_laps0		;Number of laps for first track
			ld	hl,championship_personal+6	;Point to number of laps of first track
			ld	a,(person_maps)		;Get number of personalized tracks
			ld	b,a			;Keep it in B
updatper_loop1:		ld	a,(hl)			;Get number of laps
			add	a,CHR_0			;Turn it into character
			ld	(de),a			;Store in message
			inc	hl			;Next track
			inc	de			;Next message position
			djnz	updatper_loop1		;Repeat for all tracks
			ld	a,(person_maps)		;Get number of personalized tracks
			ld	b,a			;Keep it in B
			ld	a,8
			sub	b			;Get 8-number of tracks
			jr	z,updatper_nozeros	;If all tracks are used, no need to put spaces
			ld	b,a
			xor	a			;Blank space
updatper_notracks:	ld	(de),a			;Set space in unused tracks
			inc	de			;Next message position
			djnz	updatper_notracks	;Repeat for the rest of tracks

updatper_nozeros:	ld	hl,championship_personal+1	;Point to scores per position
			ld	b,3			;First three scores
updatper_loop2:		ld	a,(hl)			;Get score
			add	a,CHR_0			;Turn into a character
			ld	(de),a			;Store in message
			inc	hl			;Next score
			inc	de			;Next message position
			djnz	updatper_loop2		;Do it for 3 scores
			inc	de			
			inc	de			;Third change here
			ld	a,(hl)			;Get points for 4th position
			add	a,CHR_0			;Turn into a character
			ld	(de),a			;Store in message
			inc	hl			;Next score
			inc	de			;Next message position
			ld	a,(hl)			;Get points for fast lap
			add	a,CHR_0			;Turn into a character
			ld	(de),a			;Store in message
			ld	bc,mpers_param
			jp	printmessage_nr		;Print message and return

;******
;menu2: Display the credits
;******
menu2:			ld	hl,screencredits
			ld	de,screenbuffer
			call	deexo			;Uncompress credits screen
			ld	hl,1860h		;Don't touch the first three lines (logo+blank line)
			ld	bc,672
			ld	de,screenbuffer
			call	ldrvrm			;Credits screen ready...
			call	pmess_select_center	;Center text
			ld	bc,message_credits
			call	printmessage_nr		;Print credits message
			ld	bc,200			;Counter
menu2_loop:		push	bc
			halt
			halt
			halt				;Wait 3 frames
			call	check_key		;Check if a key has been pressed
			pop	bc
			jp	nz,menuprint2		;If a key was pressed, go back to menu
			dec	bc
			ld	a,b
			or	c
			jr	nz,menu2_loop		;Keep looping for 12 seconds (@50 Hz), then go to Records part

;******
;menu3: Display the fast lap records for all tracks and categories. If any record is shared, the colors will be swapping.
;******
menu3:			ld	hl,mapssmall
			ld	de,mapa
			call	deexo			;Uncompress micromaps
			ld	hl,1400h
			ld	de,mapa
			ld	bc,1024
			call	ldrvrm			;128-255: micromaps chars

			ld	hl,screenrecords
			ld	de,screenbuffer+96
			call	deexo			;Uncompress records screen
			ld	hl,screenbuffer+96+544
			ld	a,128			;First character
			ld	c,4			;4 lines
menu3_putmimaps_outer:	ld	b,32
menu3_putmimaps_inner:	ld	(hl),a			;Set character
			add	a,4			;Point to next character
			inc	hl
			djnz	menu3_putmimaps_inner	;Do it for 32 characters per line
			sub	127			;Point to next row of character
			dec	c
			jr	nz,menu3_putmimaps_outer ;Loop for all lines
			
			ld	hl,1860h		;Don't touch the first three lines (logo+blank line from menu)
			ld	bc,672			;Number of bytes to send to VRAM
			ld	de,screenbuffer+96
			call	ldrvrm			;Credits screen ready in VRAM
			ld	bc,mrecords_fixed
			call	printmessage_nr		;Print records fixed message

			ld	hl,mapas+6
			ld	de,screenbuffer+256+2	;Place to put first record
menu3_numbers_outer:	ld	b,4			;Number of categories
menu3_numbers_inner:	push	bc
			push	hl
			ld	a,(hl)
			inc	hl
			ld	l,(hl)
			ld	h,a			;Get record time in HL
			call	timeprint3		;Print record into time message
			ld	bc,6
			ldir				;Print it
			pop	hl
			inc	hl
			inc	hl			;Point to next record
			inc	e
			inc	de			;Point to next records in screen buffer
			pop	bc
			djnz	menu3_numbers_inner	;Do it for all 4 categories
			ld	bc,8
			add	hl,bc			;Point to next track
			ld	a,e
			cp	2
			jr	nz,menu3_numbers_outer	;Do it for all tracks

			ld	de,mapa			;place to create attribute maps
			ld	a,12			;Number of attribute maps to create

menu3_copymaps:		ld	hl,screenbuffer+256	;Records start
			ld	bc,256			;Number of bytes to copy
			ldir				;Copy from records to current map
			dec	a
			jr	nz,menu3_copymaps	;Do it 12 times

			ld	hl,mapas+2		;Point to record holder of first map
			ld	de,mapa+3		;Point to first attribute map
menu3_colors_outer:	ld	b,4			;4 car colors to test
menu3_colors_inner:	ld	a,(hl)			;Who got the record?
			and	a
			jr	z,menu3_colors_white	;If 0, nobody set a record yet -> leave it white
			push	bc
			push	hl
			push	de
			ld	b,4
			ld	ix,mapa_tablecolors	;colors table
			ld	l,0			;Number of colors counter
			ld	h,a			;H has the records holders now
			xor	a			;First colorset
menu3c_find:		add	a,24			;Try next colorset
			rrc	h			;Next is a recordholder?
			jr	nc,menu3c_notthis
			bit	3,h			;Human or not?
			jr	z,menu3c_notbright
			add	a,96			;If human, set bright color
			ld	(ix),a			;Set color in table
			sub	96			;Substract color again
			jr	menu3c_putted
menu3c_notbright:	ld	(ix),a			;Set color in table
menu3c_putted:		inc	ix			;Next position in table
			inc	l			;Increase number of colors
menu3c_notthis:		djnz	menu3c_find		;Test all car colors
					;Here: L=number of colors, table in ix
			ld	c,12			;Number of color maps
menu3c_outer2:		ld	ix,mapa_tablecolors
			ld	h,l			;H = counter
menu3c_middle2:		ld	b,5			;5 chars to set color
menu3c_paint:		ld	a,(de)			;Get current color in (DE)
			and	a			;If not 0...
			jr	z,menu3c_noaddtospace
			add	a,(ix)			;Add color from table
			ld	(de),a			;Paint map with color
menu3c_noaddtospace:	inc	e			;Next position
			djnz	menu3c_paint		;Do it for all 5 characters
			ld	a,e
			sub	5
			ld	e,a			;Restore position
			inc	d			;Next color map
			dec	c
			jr	z,menu3c_exithere	;Exit if all 12 color maps have been filled
			inc	ix			;Next color from the table
			dec	h			;If we've done all colors from the table
			jr	nz,menu3c_middle2
			jr	menu3c_outer2		;Restart from the first again
menu3c_exithere:	pop	de
			pop	hl
			pop	bc
menu3_colors_white:	inc	hl			;Next category

			ld	a,e
			add	a,8
			ld	e,a			;Next screen position
			djnz	menu3_colors_inner	;Loop for all categories
			jr	c,menu3_doloop		;If carry is set, we've just finished the last, so exit here
			ld	bc,12
			add	hl,bc			;Advance to next track
			jr	menu3_colors_outer	;Repeat till all records are done

menu3_doloop:		ld	bc,200			;Main counter
			ld	de,mapa			;First map color
menu3_loop:		push	bc
			push	de
			halt
			halt
			halt				;Pause of 3 frames
			call	check_key		;Check for a key
			pop	de
			jr	nz,menu_toreprint	;If a key is pressed, go back to menu
			ld	hl,1900h		;screen, 2nd third in VRAM
			ld	bc,256			;Number of bytes to dump
			push	de
			call	ldrvrm			;Dump the color map into the second third attributes
			pop	de
			inc	d			;Next color map
			ld	a,d
			cp	(mapa/256)+12		;Final?
			jr	nz,menu3_loop_noback
			ld	de,mapa			;Yes: Start again from the first
menu3_loop_noback:	pop	bc
			dec	bc			;Decrease counter
			ld	a,b
			or	c
			jr	nz,menu3_loop		;Keep looping
			jp	menuprint2		;Go back to menu
			
menu_toreprint:		pop	bc			;Discard value in stack
			jp	menuprint2		;Go back to menu

;***********
;start_game: Start the whole championship
;***********
start_game:		pop	bc			;Discard menu counter value in stack
			di				;Disable interrupts
			call	silencia_ay		;Silence the PSG
			call	cls			;Clear screen
			ld	bc,4*256+4		;B = 4 (to loop all cars), C = 4 (number of running cars)
			ld	ix,car1			;First car
			ld	iy,camera_left		;Left camera
			xor	a
			ld	(humans),a		;Reset number of humans
			ld	(cpus),a		;Reset number of CPUs
			ld	(currentmap),a		;Reset current map number
			ld	(aborted),a		;Reset aborted state
sg_preparecar:		ld	a,mapa_checks/256
			ld	(ix+SP_NOVICE),a	;Default checkpoints, high part of address
			ld	a,(ix+SP_CONTOPT)	;Get control option
			cp	9
			jr	z,sg_outofrace		;Skip if it's out of race
			ld	hl,control_key1		;Control for keyboard 1
			and	a
			jr	z,sg_setcontrol		;Set human control if CONTOPT==0
			ld	hl,control_key2		;Control for keyboard 2
			dec	a
			jr	z,sg_setcontrol		;Set human control if CONTOPT==1
			ld	hl,control_key3		;Control for keyboard 3
			dec	a
			jr	z,sg_setcontrol		;Set human control if CONTOPT==2
			ld	hl,control_joy1		;Control for joystick 1
			dec	a
			jr	z,sg_setcontrol		;Set human control if CONTOPT==3
			ld	hl,control_joy2		;Control for joystick 2
			dec	a
			jr	z,sg_setcontrol		;Set human control if CONTOPT==4
			ld	hl,control_kemp		;(Control for joystick kempston, disabled on MSX)
			dec	a
			jr	z,sg_setcontrol		;Set human control if CONTOPT==5
			ld	hl,96*256+80		;H=96, L=80, D=64 for EASY CPU
			ld	d,64
			dec	a			
			jr	z,sg_setcpucontrol	;Set CPU control if CONTOPT==6
			ld	hl,112*256+96		;H=112, L=96, D=16 for NORMAL CPU
			ld	d,16
			dec	a
			jr	z,sg_setcpucontrol	;Set CPU control if CONTOPT==7
			ld	hl,128*256+128		;H=128, L=128, D=0 for HARD CPU
			ld	d,0
							;Set CPU control if CONTOPT==8
sg_setcpucontrol:	ld	(ix+SP_IARAND1),h
			ld	(ix+SP_IARAND2),l
			ld	(ix+SP_IARAND3),d	;Fill AI constants
			ld	hl,control_varied	;Control routine
			ld	a,(cpus)
			inc	a
			ld	(cpus),a		;Increase number of CPU controlled cars
			jr	sg_nothuman		
sg_setcontrol:		ld	a,(humans)
			inc	a
			ld	(humans),a		;Increase number of human controlled cars
			ld	a,(mode)
			and	1
			add	a,(ix+SP_NOVICE)
			ld	(ix+SP_NOVICE),a	;If category is JUNIOR or GP2, use novice checkpoints instead
sg_nothuman:		ld	(ix+SP_CONTROL+1),h
			ld	(ix+SP_CONTROL),l	;Set control routine
			ld	(ix+SP_CONTROLBAK+1),h
			ld	(ix+SP_CONTROLBAK),l	;Set backup control routine
sg_finaltouchs:		ld	a,(ix+SP_MASTCOL)
			ld	(ix+SP_COLOR),a		;Restore color
			ld	a,(ix+SP_MASTCOLORMARKER)
			ld	(ix+SP_COLORMARKER),a	;Restore marker color
			xor	a			;Initial state at 0
			jr	sg_nextcar
sg_outofrace:		dec	c			;Decrease number of running cars
			ld	a,17			;Black
			ld	(ix+SP_COLOR),a		;Set color
			ld	a,131+6*16+6
			ld	(ix+SP_COLORMARKER),a	;Black colorset
			ld	a,16			;Bit 4 of state high > Don't race
sg_nextcar:		ld	(ix+SP_STATE),a		;Set initial state
			xor	a
			ld	(ix+SP_POINTS),a	;Clear points
			ld	de,car2-car1
			add	ix,de			;Point to next car
			dec	b
			jp	nz,sg_preparecar	;Loop for all 4 cars
			ld	a,c
			ld	(nactive_cars),a	;Set number of active cars
			
			ld	a,(mode)		;Get category
			cp	2
			ld	a,131+0*8+5		;color for time and number of laps
			jr	c,sg_highspeed
			dec	a			;color for compass (1 last character colored, so that character remains with compass color)
sg_highspeed:		ld	(lap_left_ncol),a	;Set that color in message for left side
			ld	(lap_right_ncol),a	;Set that color in message for right side
			ld	a,(mode)		;Prepare speeds according to category
			add	a,a
			add	a,a
			add	a,a
			add	a,a
			ld	l,a
			ld	h,0
			ld	de,spd_0		;Base table
			add	hl,de			;Once the address of speed table that we're going to use is calculated...
			push	hl
			ld	de,speeds_base		;Take it out
			ld	bc,16
			ldir				;...copy the 16 bytes to the speed table that will be used in the game
			pop	hl
			ld	bc,6
			ldir				;Repeat 6 first bytes from that table
							;Start stage entrance point

;************
;start_stage: Start the next stage
;************
start_stage:		call 	vdpdis			;Disable VDP
			call	cls			;Clear screen
			call	pmess_select_center	;Maybe this doesn't matter now
			;call	wait_key		;Wait for key
			ld	a,tablepoints%256
			ld	(tablepointer),a
			xor	a
			ld	(tableadvance),a	;Default no advance in tablepoints
			ld	(positionwin),a 	;Reset position counter to 0
			ld	(minimap_cycle),a 	; and minimap_cycle too
			ld	bc,4*256+2		;B=4 (number of cars), C=2 (number of cameras)
			ld	ix,car1			;First car
			ld	iy,camera_left		;Left camera
setcamloop:		ld	e,(ix+SP_CONTOPT)	;Get control option
			ld	a,5
			cp	e			;Below 5?
			jr	c,setcam_nohum
			ld	a,(ix+SP_CAMERA)
			ld	(iy+CAM_CAR),a		;Set camera
			set	7,(ix+SP_STATE)		;Set human bit
			dec	c
			ld	iy,camera_right
setcam_nohum:		ld	de,car2-car1
			add	ix,de			;Point to next car
			djnz	setcamloop		;Loop to set cameras to point cars controlled by humans
			ld	a,c
			and	a
			jr	z,camerasallset		;If all cameras are set, skip to next part

			ld	b,4			;4 cars
			ld	ix,car1			;First car
setcamloop2:		ld	e,(ix+SP_CONTOPT)	;Get control option
			ld	a,e
			cp	6
			jr	c,setcam_nocpu		;Below 6, it ain't CPU
			cp	9
			jr	z,setcam_nocpu		;Neither is 9
			ld	a,(ix+SP_CAMERA)
			ld	(iy+CAM_CAR),a		;Set camera
			dec	c
			ld	iy,camera_right		;Next camera to set
setcam_nocpu:		ld	de,car2-car1
			add	ix,de			;Point to next car
			djnz	setcamloop2		;Repeat loop for all cars

camerasallset:		ld	hl,0
			ld	(main_ticks),hl		;Reset main timer
			ld	(lastcamchange),hl	;Also reset last camera change time
			ld	(lastsndchange),hl	;And last sound change time too
			ld	a,CHR_0_
			ld	(message_cl_m1),a	;Set 0 to clock in minute units
			ld	hl,message_cl_m10
			ld	a,CHR_0
			ld	(hl),a			;Set 0 to clock in minute tens
			inc	hl
			inc	hl
			ld	(hl),a			;and second tens
			inc	hl
			ld	(hl),a			;and second units
			di
			
			ld	a,(currentmap)
			and	3			;2 bit, up to 4 songs
			call	CARGA_CANCION		;Load corresponding song in WYZPlayer
			ld	hl,isr_wyz
			ld	(interrupt_redir),hl	;Set interrupt routine
			ld	a,(recording)
			cp	1
			call	z,record_preparerecord	;If we can, prepare to record
			ld	a,(championship)	;Get championship number
			add	a,a
			ld	l,a
			ld	h,championships_data/256
			ld	a,(hl)			;Get championship address
			inc	l			
			ld	h,(hl)			
			ld	l,a			;HL points to championship
			ld	a,(hl)			;Get number of tracks
			add	a,16
			ld	(mfirst_total),a	;Set total number of championship tracks in the first message in marker
			inc	l			;Point to scores per positions
			ld	bc,5
			ld	de,tablepoints
			ldir				;Copy scores table, now HL points to the number of laps for first track
			ld	a,(currentmap)		;Get current track inside the championship
			add	a,l
			ld	l,a			;HL points to number of laps in current track
			ld	a,(currentmap)
			add	a,17
			ld	(mfirst_nrace),a	;Set number of track in first message in marker
			ld	a,(hl)			;Check the number of laps
			and	a
			jp	z,endgame		;End game if it is the end of championship
			ld	(current_stage_laps),a
			add	a,16
			ld	(mfirst_nlaps),a	;Set number of laps to race in the first message in marker
			ld	a,8
			add	a,l
			ld	l,a			;HL now points to track index
			ld	c,(hl)			;Get track number*2
			ld	a,c
			add	a,a
			add	a,a
			add	a,a
			ld	c,a			;C = track number * 16
			ld	hl,mapas
			add	hl,bc			;Address of current map in table mapas
			ld	b,(hl)
			inc	hl
			ld	h,(hl)
			ld	l,b			;Get compressed map address to uncompress it
			push	hl
							;Prepare tiles and markers...
			ld	de,tilesunpacked
			ld	hl,0
			ld	bc,136*8		;Copy chars 000-135 to first third
			call	ldrvrm
			ld	hl,800h
			ld	bc,106*8		;Copy chars 000-105 to second third
			ld	de,tilesunpacked
			call	ldrvrm
			ld	hl,1000h
			ld	bc,131*8		;Copy chars 000-130 to third third
			ld	de,tilesunpacked
			call	ldrvrm
			ld	hl,1000h+8*131
			ld	de,tilesunpacked+8*136
			call	ldrvrmbyte		;Copy char 136 to third third (to 131)
			ld	de,tilesunpacked+8*137
			ld	hl,8*201
			ld	bc,8*23			;Copy chars 137-159 to first third (201-223)
			call	ldrvrm		
			ld	de,tilesunpacked+8*137
			ld	hl,1000h+8*201
			ld	bc,8*23			;Copy chars 137-159 to third third (201-223)
			call	ldrvrm
			
			ld	de,colorsunpacked
			ld	hl,2000h
			ld	bc,136*8		;Copy colors 000-135 to first third
			call	ldrvrm
			ld	de,colorsunpacked
			ld	hl,2800h
			ld	bc,106*8		;Copy colors 000-105 to second third
			call	ldrvrm
			ld	de,colorsunpacked
			ld	hl,3000h
			ld	bc,131*8		;Copy colors 000-130 to third third
			call	ldrvrm
			ld	hl,3000h+8*130
			ld	b,59			;Number of minimap chars
greenmap:		push	bc
			ld	bc,8			;Copy color 136 to third third (to 131-189)
			add	hl,bc
			push	hl
			ld	de,colorsunpacked+8*136
			call	ldrvrmbyte		;Copy a byte
			pop	hl
			pop	bc
			djnz	greenmap		;Do it for all minimap chars
			ld	de,colorsunpacked+8*137
			ld	hl,2000h+8*201
			ld	bc,8*23			
			call	ldrvrm			;Copy color 137-159 to first third (201-223)
			ld	de,colorsunpacked+8*137
			ld	hl,3000h+8*201
			ld	bc,8*23			
			call	ldrvrm			;Copy color 137-159 to third third (201-223)
			ld	de,colorsunpacked+8*160
			ld	hl,2000h+8*190
			ld	bc,8*11			
			call	ldrvrm			;Copy color 160-170 to first third (190-200)
			ld	de,colorsunpacked+8*160
			ld	hl,3000h+8*190
			ld	bc,8*11			
			call	ldrvrm			;Copy color 160-170 to third third (190-200)
			ld	de,colorsunpacked+8*160
			ld	hl,2000h+8*136
			ld	bc,8*4			
			call	ldrvrm			;Copy color 160-163 to first third (136-139)
			ld	hl,screenpacked
			ld	de,screenbuffer
			call	deexo			;Uncompress ingame screen with markers

			pop	hl
			call	crearmapa		;Create the map and fix the parameters
			call	minimap_define_chars	;Define minimap characters

			ld	hl,1000h+8*132
			ld	bc,57*8
			call	ldrvrmbuffer		;Draw minimap

			xor	a
			ld	(fastforward),a		;Disable fast-forward
			ld	(humansarrived),a	;Reset counter of humans arrived
			ld	hl,spritesbuffer
			ld	(hl),a
			ld	de,spritesbuffer+1
			ld	bc,127
			ldir				;Clean sprites zone
			ld	a,3			;First time
			ld	(changemarker),a	;Signal that the marker should be updated for first time

			ld	hl,1800h
			ld	bc,768
			ld	de,screenbuffer
			call	ldrvrm			;Dump initial screen to VRAM
			call 	vdpena			;Enable VDP: show screen
			ei				;Enable interrupts

			ld	a,(msxHZ)
			cp	6
			jr	nz,mainloop
			call	dofirststepsframe	;60Hz, do first steps for first frame
			halt
							;Main loop entrance point
							
;*********
;mainloop: Main game loop, should be repeated every 5/6 frames (in 50/60Hz), until the game is aborted or the game ends.
;*********
mainloop:		xor	a
			ld	(int_ticks),a		;Reset ticks counter
			ld	a,(recording)
			cp	2			;Replay?
			jr	nz,main_nocheckendrecord
			ld	hl,(max_rec_addr)	;Yes:
			ld	de,(current_rec_addr)	;check if we've reached the end of recording
			sbc	hl,de
			jp	c,end_stage		;If we've reached the end, go to end_stage

main_nocheckendrecord:	call	control_key4		;Update extra keys
			ld	a,c
			ld	(extra_keys),a
			and	4			;Check sound key
			jr	nz,main_nochangesnd	;No sound key pressed, no change on sound
			ld	de,(lastsndchange)	;Check when the last sound change was
			ld	hl,(main_ticks)
			and	a			;No carry
			sbc	hl,de			;Compare current time to time of last sound change
			ld	a,h
			and	a
			jr	nz,main_chgsnd_ok	;If difference is over 256, it's ok
			ld	a,l
			cp	6			;Difference over 5?
			jr	c,main_nochangesnd	;No: Wait at least 0.6 secs to make another sound change
							;Yes: ok
main_chgsnd_ok:		ld	(lastsndchange),hl	;Update last sound change timestamp -> BUG: the substracted value needs to be added back
			ld	a,(current_sound)	;Get current_sound variable
			inc	a			;Increase current_sound
			cp	6			;Check if we reached the limit
			jr	nz,main_storesound
			ld	a,4
			ld	(snd_state),a		;Disable keyclick sound
			xor	a
			ld	(snd_speed),a		;Disable keyclick sound
main_storesound:	ld	(current_sound),a	;Update current_sound variable
main_nochangesnd:	ld	a,(msxHZ)
			cp	5
			call	z, dofirststepsframe	;Do the first steps if we're at 50Hz

			ld	a,(minimap_cycle)
			inc	a
			and	7
			ld	(minimap_cycle),a	;Update minimap cycle variable
	
							;Reset relative positions
			ld	a,1
			ld	(car1+SP_RELPOS),a
			ld	(car2+SP_RELPOS),a
			ld	(car3+SP_RELPOS),a
			ld	(car4+SP_RELPOS),a
	
							;Check if there is any collision, and recreate relative positions
			ld	ix,car1
			ld	iy,car2
			call	checkcollision		;Check for collisions & set position between cars 1 and 2
			ld	iy,car3
			call	checkcollision		;Check for collisions & set position between cars 1 and 3
			ld	iy,car4
			call	checkcollision		;Check for collisions & set position between cars 1 and 4
			ld	ix,car3
			call	checkcollision		;Check for collisions & set position between cars 3 and 4
			ld	ix,car2
			call	checkcollision		;Check for collisions & set position between cars 2 and 4
			ld	iy,car3
			call	checkcollision		;Check for collisions & set position between cars 2 and 3

			ld	iy,camera_left		;Select left camera
			ld	a,(extra_keys)
			and	2
			call	z,switchcar		;If the CAM 1 key is pressed, switch car at the left half
			ld	a,(iy+CAM_CAR)
			ld	(other),a
			ld	iy,camera_right
			ld	a,(extra_keys)
			and	1
			call	z,switchcar		;If the CAM 2 key is pressed, switch car at the right half
			ld	a,(iy+CAM_CAR)
			ld	(other),a
			ld	a,(extra_keys)
			and	8
			call	z,pause			;If the PAUSE key is pressed, call the pause procedure
			ld	iy,camera_left
			ld	a,(fastforward)		;Check fastforward variable
			cp	4
			jr	z,fastfowardingexception
			and	a
			jr	nz,fastforwarding	;If fastforward variable is not 0 or 4, jump screen update
			
fastfowardingexception:	ld	hl,buffer
			ld	(carbuffer),hl		;Init the cars buffer
			ld	iy,camera_left
			
			call	keyclick_sound		;Make keyclick sound
			
			call	volcarmapa		;Left camera: put the map addresses in the cameras
			call	printbg			;Paint the background in the buffer
			call	printcarsleft		;Put visible car characters on the table

			ld	iy,camera_right

			call	volcarmapa		;Right camera: put the map addresses in the cameras
			call	printbg			;Paint the background in the buffer
			call	printcarsright		;Put car characters on the table

			call	mixchars		;Mix cars characters with background
			
			call	keyclick_sound		;Make keyclick sound again
			
			call	mixcolors		;Mix cars colors with background
			call	preparesprites		;Do all sprites list, later
			ld	a,208
			ld	(iy),a			;Mark last sprite
			call	assignchars		;Assign new chars to their tile numbers, write those tile numbers in the buffer
						;Right here:
						;C-232 (if >0) == number of 1st third chars to write to VRAM in DUMP
						;E-224 (if >0) == number of 3rd third chars to write to VRAM in DUMP
			ld	b,0
			ld	a,c
			dec	a
			sub	231			;Number of 1st third chars to write to VRAM in DUMP
			jr	nc,charsfirstthird
			xor	a			;0 chars
charsfirstthird:	add	a,a
			add	a,a
			add	a,a			;*8
			ld	c,a			;BC = length of buffers to dump to 1st third
			ld	(duration_1st),bc	;Store that lenght
			ld	hl,(carbuffer)
			add	hl,bc
			ld	(carbuffer3),hl		;Store start position of 3rd third chars buffer
			ld	hl,(colorbuffer)
			add	hl,bc
			ld	(colorbuffer3),hl	;Store start position of 3rd third color buffer
			ld	a,e
			sub	224			;Number of 3rd third chars to write to VRAM in DUMP
			add	a,a
			add	a,a			;*4
			ld	l,a
			ld	h,0
			add	hl,hl			;*8 in HL
			ld	(duration_3rd),hl	;Store length of buffers to dump to 3rd third 

				;Right here: for the dump, other than all active area, we must dump (duration_1st) bytes, starting from (carbuffer)
				;to 1st third character 232 (VRAM: 232*8), another (duration_1st) bytes, starting from (colorbuffer) to 1st third
				;color 232 (232*8+2000h), (duration_3rd) bytes, starting from (carbuffer3) to 3rd third character 224 (224*8+1000h),
				;and finally (duration_3rd) bytes, starting at (colorbuffer3) to 3rd third color 224 del (224*8+3000h)
			call	dump			;Dump screen
			
fastforwarding:		ld	a,(fastforward)		;get fastforward state
			and	a
			jr	z,notfastforwarding
			dec	a			;Decrease state
			jr	nz,norecharge
			ld	a,10			;from 0 to 10
norecharge:		ld	(fastforward),a		;Update state

			cp	3
			jr	z,notfastforwarding	;fastforward==3?
			ld	a,(msxHZ)		;Yes: check retraze speed
			cp	6			;60Hz?
			call	z, dofirststepsframe	;call to dofirststepsframe, because next frame won't be fastforwarded

notfastforwarding:	ld	hl,(main_ticks)		;Get total time
			ld	iy,message_clock-CAM_MESSTIME-1
			call	timeprint		;Print it into main clock message
			ld	bc,message_clock
			call	printmessage		;Print main clock message
			ld	bc,camera_left+CAM_MESSTIME
			call	printmessage		;Print left car time
			ld	bc,camera_right+CAM_MESSTIME
			call	printmessage		;Print right car time

			ld	hl,(main_ticks)
			inc	hl
			ld	(main_ticks),hl		;Increase time counter

			ld	a,(changemarker)	;Check if we must print or change the marker
			and	a
			jp	z,mainloop		;No: Back to repeat the main loop

			cp	4			;Changermarker==4?
			jr	nz,mark_nodelay
			ld	a,1			;Yes: New value = 1
			ld	(changemarker),a	
			jp	mainloop		;Repeat the main loop
mark_nodelay:		cp	3
			jr	nz,mark_nofirstmessage
			ld	a,(recording)		;If value is 3, we need to print the first message
			cp	2			;Are we replaying?
			ld	bc,mfirst_play
			jr	nz,mark_firstmessage	;  No: go to print the first play message
			ld	bc,mfirst_replay	;  Yes: go to print the first replay message
mark_firstmessage:	call	printmessage
			ld	bc,mfirst_all		;Common part to first messages
			call	printthemarker2		;Print it
			jp	mainloop		;Repeat the main loop
mark_nofirstmessage:	dec	a
			jr	nz,doprintthemarker	;If value was 1, go to print the marker
			call	preparemarker		;If value was 2, prepare the marker
			jp	mainloop		;Repeat the main loop
doprintthemarker:	call	printthemarker		;Print the marker
			jp	mainloop		;Repeat the main loop

;**************
;preparemarker: Prepare the marker to be updated on next frame.
;**************
preparemarker:		ld	ix,2*car1-car2		;First car address minus the lenght of its variables space
			ld	de,car2-car1
mark_buc1:		add	ix,de			;Point to next car
			ld	a,(ix+SP_RELPOS)
			cp	4
			jr	nz,mark_buc1		;Keep looking until we find the first position
			ld	a,(ix+SP_COLORMARKER)
			ld	(message_col1st),a	;Set color
			ld	de,message_col1st+1	;Fixed position

			ld	hl,mes_mark_1st		;HL points to "WINNER"
			bit	2,(ix+SP_STATE)		;Check if we've finished
			jr	nz,mark_winner1		;Finished: Go to print HL's message.
			
							;Not finished yet: print current lap and checkpoint
			ld	a,(ix+SP_TIMESTAMPLAP)	;Get lap of the last timestamp
			add	a,16			;Turn into character
			ld	(mes_ml_lap),a		;Store lap in message
			ld	a,(ix+SP_TIMESTAMPCP)	;Get checkpoint
			rlca
			rlca
			rlca				;Divide by 8
			and	7			;Keep only 3 bits
			add	a,16			;Turn into character
			ld	(mes_ml_ch1),a		;Store checkpoint number
			ld	hl,mes_mark_lapch	;HL points to Lx-CHy message
mark_winner1:		ld	bc,6
			ldir				;Print first message (WINNER/Lx-CHY)
			inc	de			;Point to next mark_message writing position
			push	ix			;Store last car variables address
			pop	iy			;Last car variables address in IY now
			push	de
			ld	ix,2*car1-car2		;First car address minus the lenght of its variables space
			ld	de,car2-car1
mark_buc2:		add	ix,de			;Point to next car
			ld	a,(ix+SP_RELPOS)
			cp	3			;Look for second position now
			jr	nz,mark_buc2
			pop	de			;DE=mark_message writing position here
		
			ld	a,(ix+SP_TIMESTAMPLAP)
			cp	(iy+SP_TIMESTAMPLAP)	;lap number different between first and second?
			jr	nz,mark_separate2	;Yes -> go to print a separation line between them

			ld	a,(ix+SP_TIMESTAMPCP)	;No ->
			cp	(iy+SP_TIMESTAMPCP)	;  checkpoint number different between first and second?
			jr	nz,mark_separate2	;  Yes -> go to print a separation line between them
							;  No -> print the difference
			ld	a,146
			ld	(de),a			;Place second car's line into line 2 of marker
			inc	de			;Now point to attribute byte

			ld	a,176
			ld	(message_black1),a	;Black color in 7th row

			ld	a,(ix+SP_COLORMARKER)	;Get color for second car
			ld	(de),a			;Set color
			inc	de			;Point to message

			ld	hl,mes_mark_2nd		;HL points to "SECOND"
			bit	2,(ix+SP_STATE)		;Check if we've finished
			jr	nz,mark_winner2		;Finished: Go to print HL's message
							;Not finished: print time difference

			ld	h,(ix+SP_TIMESTAMP)	
			ld	l,(ix+SP_TIMESTAMP+1)	;Get second car's timestamp
			ld	b,(iy+SP_TIMESTAMP)
			ld	c,(iy+SP_TIMESTAMP+1)	;Get first car's timestamp
			sbc	hl,bc			;Get the difference
			call	timeprint2		;Print it
			jr	mark_winner2		;Continue with third car

mark_separate2:		ld	a,146
			ld	(message_black1),a 	;Black in second row

			ld	a,152
			ld	(de),a			;Second car's line in 3rd marker row
			inc	de			;Point to color

			ld	a,(ix+SP_COLORMARKER)
			ld	(de),a			;Set color
			inc	de			;Point to message
			push	de

			ld	a,(ix+SP_TIMESTAMPLAP)	;Get lap of the last timestamp
			add	a,16			;Turn into character
			ld	(mes_ml_lap),a		;Store lap in message
			ld	a,(ix+SP_TIMESTAMPCP)	;Get checkpoint
			rlca
			rlca
			rlca				;Divide by 8
			and	7			;Keep only 3 bits
			add	a,16			;Turn into character
			ld	(mes_ml_ch1),a		;Store checkpoint in message
			ld	hl,mes_mark_lapch	;HL points to Lx-CHy message
			push	ix
			pop	iy			;Now the second car is pointed by IY
			pop	de
mark_winner2:		ld	bc,6
			ldir				;Print second message (SECOND/Lx-CHY/+x:yy.z)
			inc	de			;Point to third car's line

			push	de
			ld	ix,2*car1-car2		;First car address minus the lenght of its variables space
			ld	de,car2-car1
mark_buc3:		add	ix,de			;Point to next car
			ld	a,(ix+SP_RELPOS)
			cp	2			;Third?
			jr	nz,mark_buc3
			pop	de

			ld	a,(ix+SP_TIMESTAMPLAP)	
			cp	(iy+SP_TIMESTAMPLAP)	;lap number different between second and third?
			jr	nz,mark_separate3	;Yes -> go to print a separation line between them

			ld	a,(ix+SP_TIMESTAMPCP)	;No ->
			cp	(iy+SP_TIMESTAMPCP)	;  checkpoint number different between first and second?
			jr	nz,mark_separate3	;  Yes -> go to print a separation line between them
							;  No -> print the difference

			ld	a,(message_pos2nd)	;Get second car's line position
			add	a,6
			ld	(de),a			;Set third car's line position in next line
			inc	de
			add	a,18
			ld	(message_black2),a	;Black line 3 lines below the third's

			ld	a,(ix+SP_COLORMARKER)	;Get color
			ld	(de),a			;Set color
			inc	de			;Point to message

			ld	hl,mes_mark_3rd		;HL points to "THIRD "
			bit	2,(ix+SP_STATE)
			jr	nz,mark_winner3		;Print it if it has already arrived
			ld	h,(ix+SP_TIMESTAMP)
			ld	l,(ix+SP_TIMESTAMP+1)	;Get third car's timestamp
			ld	b,(iy+SP_TIMESTAMP)
			ld	c,(iy+SP_TIMESTAMP+1)	;Get second car's timestamp
			sbc	hl,bc			;Get the difference
			call	timeprint2		;Print it in message
			jr	mark_winner3		;Keep processing

mark_separate3:		ld	a,(message_pos2nd)	;Get second's car line position
			add	a,6
			ld	(message_black2),a	;Black color one line below
			add	a,6
			ld	(de),a			;And third car's line another below
			inc	de			;Point to color

			ld	a,(ix+SP_COLORMARKER)	;Get color
			ld	(de),a			;Set color
			inc	de			;Point to message
			push	de

			ld	a,(ix+SP_TIMESTAMPLAP)	;Get lap of the last timestamp
			add	a,16			;Turn into character
			ld	(mes_ml_lap),a		;Store lap in message
			ld	a,(ix+SP_TIMESTAMPCP)	;Get checkpoint of the last timestamp
			rlca
			rlca
			rlca				;Divide by 8
			and	7			;Keep only 3 bits
			add	a,16			;Turn into character
			ld	(mes_ml_ch1),a		;Store checkpoint in message
			ld	hl,mes_mark_lapch	;HL points to Lx-CHy message
			push	ix
			pop	iy			;IY = IX = third car
			pop	de
mark_winner3:		ld	bc,6
			ldir				;Print third message (THIRD /Lx-CHY/+x:yy.z)
			inc	de			;Point to next one

			push	de
			ld	ix,2*car1-car2		;First car address minus the lenght of its variables space
			ld	de,car2-car1
mark_buc4:		add	ix,de			;Point to next car
			ld	a,(ix+SP_RELPOS)
			cp	1			;Look for the fourth car
			jr	nz,mark_buc4
			pop	de

			ld	a,(ix+SP_TIMESTAMPLAP)
			cp	(iy+SP_TIMESTAMPLAP)	;lap number different between third and fourth?
			jr	nz,mark_separate4	;Yes -> go to print a separation line between them
			ld	a,(ix+SP_TIMESTAMPCP)	;No ->
			cp	(iy+SP_TIMESTAMPCP)	;  checkpoint number different between first and second?
			jr	nz,mark_separate4	;  Yes -> go to print a separation line between them
							;  No -> print the difference

			ld	a,(message_pos3rd)	;Get third car's line position
			add	a,6
			ld	(de),a			;Set fourth car's line position in next line
			inc	de
			add	a,6
			ld	(message_black3),a	;Black line 1 lines below the fourth's

			ld	a,(ix+SP_COLORMARKER)	;Get color
			ld	(de),a			;Set color
			inc	de			;Point to message

			ld	hl,mes_mark_4th		;HL points to "LOSER!"
			bit	2,(ix+SP_STATE)
			jr	nz,mark_winner4		;Print it if it has already arrived

			ld	h,(ix+SP_TIMESTAMP)
			ld	l,(ix+SP_TIMESTAMP+1)	;Get fourth car's timestamp
			ld	b,(iy+SP_TIMESTAMP)
			ld	c,(iy+SP_TIMESTAMP+1)	;Get third car's timestamp
			sbc	hl,bc			;Get the difference
			call	timeprint2		;Print it in message
			jr	mark_winner4		;Keep processing

mark_separate4:		ld	a,(message_pos3rd)	;Get third's car line position
			add	a,6
			ld	(message_black3),a	;Black color one line below
			add	a,6
			ld	(de),a			;And fourth car's line another below
			inc	de			;Point to color

			ld	a,(ix+SP_COLORMARKER)	;Get color
			ld	(de),a			;Set color
			inc	de			;Point to message
			push	de

			ld	a,(ix+SP_TIMESTAMPLAP)	;Get lap of the last timestamp
			add	a,16			;Turn into character
			ld	(mes_ml_lap),a		;Store lap in message
			ld	a,(ix+SP_TIMESTAMPCP)	;Get checkpoint of the last timestamp
			rlca
			rlca
			rlca				;Divide by 8
			and	7			;Keep only 3 bits
			add	a,16			;Turn into character
			ld	(mes_ml_ch1),a		;Store checkpoint in message
			ld	hl,mes_mark_lapch	;HL points to Lx-CHy message

			pop	de
mark_winner4:		ld	bc,6
			ldir				;Print fourth message (LOSER!/Lx-CHY/+x:yy.z)

			ld	a,2
			ld	(changemarker),a	;Signal that marker should be printed next frame
			ret

;****************
;printmessage_nr: Print message while rainbow effect is disabled
;****************
printmessage_nr:	xor	a
			ld	(rainbow_on),a		;Disable rainbow effect
			call	printmessage		;Print message
			ld	a,1
			ld	(rainbow_on),a		;Enable rainbow effect
			ret


;*************
;printmessage: Prints the message pointed to by BC. DE holds the screen address to write to. The message format is explained in the messages.asm file.
;*************
pmess_attr:		dec	a			;Check for 131
			jr	nz,pmess_attrsimple
							;131: simple attribute, repeated for 
			ld	hl,(pmess_address)	;Get VRAM address
			push	bc
			ld	a,(VDPportWRITE)	;Get VDP write port
			ld	c,a			;Port
			ld	a,l
			di				;Disable interrupts
                	out     (c),a			;Output low part of VRAM address
                	ld	a,h
			or	96			;Add color bit too!
			ei				;Enable interrupts
			out     (c),a			;Output high part of VRAM address
			pop	bc
			
			ld	a,(bc)			;Get number of characters
			inc	bc			;Next message position
			ld	e,a			;Keep it in E
			ld	a,(bc)			;Get attribute
			inc	bc			;Next message position
			
			push	bc			;Keep message position
;			ld	c,e
			ld	bc,(VDPportWRITE)	;Get VDP write port
			dec	c			;Data port
pmess_attrextend_outer:	ld	b,8			;8 bytes
pmess_attrextend_inner:	out     (c),a			;Output attribute
			djnz	pmess_attrextend_inner	;The whole character
			dec	e			
			jr	nz,pmess_attrextend_outer	;Loop for E characters
			pop	bc			;Restore message position
			jr	printmessage		;Keep processing the message

pmess_attrsimple:	push	bc			;Keep message position
			dec	a			;Here A=byte read from message-132.
							;  The 4 lower bits set the number of bytes
							;  and the 3 highest bits set the colorset to use
			ld	c,a			;Keep it in C
			and	15			;Keep lower bits
			ld	b,a			;B=num of chars
			inc	b			;+1 = 1-16
			ld	a,c
			and	112			;Isolate high bits
			rrca				;Colorset*8
			ld	e,a
			ld	d,0			;DE=colorset*8
			ld	hl,colorsets		
			add	hl,de			;Add colorsets base
			ex	de,hl			;DE=colorset address

			ld	hl,(pmess_address)	;Get VRAM address
			ld	a,(VDPportWRITE)	;Get VDP write port
			ld	c,a			;Put into C
			ld	a,l
			di				;Disable interrupts
                	out     (c),a			;Output low part of VRAM address
                	ld	a,h
			or	96			;Add color bit too!
			ei				;Enable interrupts
                	out     (c),a			;Output high part of VRAM address
			ex	de,hl			;colorset address in HL
			dec	c			;Data port
pmess_attrloop:		push	bc			;Keep character counter
			push	hl			;Keep colorset address
			ld	b,8			;8 bytes for a character
			outi
			nop
			nop
			outi
			nop
			nop
			outi
			nop
			nop
			outi
			nop
			nop
			outi
			nop
			nop
			outi
			nop
			nop
			outi
			nop
			nop
			outi				;Output 8 bytes
			pop	hl			;Restore address
			pop	bc			;Restore counter
			djnz	pmess_attrloop		;Loop all bytes
			pop	bc				;Restore message position
			jr	printmessage		;Keep processing the message

pmess_nochar:		ld	de,0			;D=0 for 1st third
			sub	128
			jr	z,pmess_position	;If 128, go set first third
			ld	d,8			;D=8 for 2nd third
			dec	a
			jr	z,pmess_position	;If 129, go set second third
			ld	d,16			;D=16 for 3rd third
			dec	a
			jp	nz,pmess_attr		;If not 130, go set attribute
							;If 130, set third third
pmess_position:		ld	a,(bc)			;Get character number
			inc	bc			;Point to next character
			ld	l,a
			ld	h,0
			add	hl,hl			;*2
			add	hl,hl			;*4
			add	hl,hl			;*8
			add	hl,de			;Add third constant
printmessage_addr:	ld	(pmess_address),hl	;Store new VRAM address
printmessage:		ld	a,(bc)			;Get message byte
			inc	bc			;Point to next address
			cp	127
			ret	z			;127=end
			jr	nc,pmess_nochar
			ld	h,charset/1024		;Charset address
			push	bc
			add	a,a			;Char code *2
			ld	l,a
			add	hl,hl			;*4
			add	hl,hl			;*8
			ex	de,hl			;Keep it in DE
			ld	hl,(pmess_address)	;Get VRAM address
			ld	a,(VDPportWRITE)	;Get VDP write port
			ld	c,a			;In register C
			ld	a,l
			di					
                	out     (c),a			;Set low part of VRAM address
                	ld	a,h
			or	64
			ei
                	out     (c),a			;Set high part of VRAM address
			ld	bc,8
			add	hl,bc			;Next char
			ld	(pmess_address),hl	;Set next VRAM address
			ld	hl,(charsrout)		;Get printchar routine
			jp	(hl)			;Go to it

pmess_return:		pop	bc
			jr	printmessage		;Keep processing the message

pmess_select_left:	ld	hl,pchar_left		;Left inclination printchar routine
			jr	pmess_changerout
pmess_select_right:	ld	hl,pchar_right		;Right inclination printchar routine
			jr	pmess_changerout
pmess_select_center:	ld	hl,putchar_down		;Centered printchar routine
pmess_changerout:	ld	(charsrout),hl		;Set selected routine
			ret

;*******
;vdpdis: Disable the VDP, to show a black screen
;*******
vdpdis:         	ld	a,(VDPportWRITE)	;Get VDP write port
			ld	c,a
			ld	a,(VDPREG)		;Get VDP register value
                	and	191			;Clear bit 6
                	ld	(VDPREG),a		;Store VDP register value
			ld	b,1			;Register number
			jr	w_regvdp		;Go to modify register

;*******
;vdpdis: Enable the VDP, to show a black screen
;*******
vdpena:         	ld	a,(VDPportWRITE)	;Get VDP write port
			ld	c,a
			ld	a,(VDPREG)		;Get VDP register value
                	or	64			;Set bit 6
                	ld	(VDPREG),a		;Store VDP register value
			ld	b,1			;Register number

;*********
;w_regvdp: Write a value to an VDP control register
;*********
w_regvdp:      		di				;Disable interrupts
                	out     (c),a			;Write data
			ld	a,b			;Get register number
                	or	128
                	ei				;Enable interrupts
                	out     (c),a			;Write register number
                	ret

;***************
;ldrvrmbuffer2k: Copy 2K of data from buffer to VRAM address indicated by HL
;***************
ldrvrmbuffer2k:		ld	bc,2048			;BC = number of bytes

;*************
;ldrvrmbuffer: Copy BC bytes of data from buffer to VRAM address indicated by HL
;*************
ldrvrmbuffer:		ld	de,buffer		;DE = start of data to be moved

;*******
;ldrvrm: Copy BC bytes of data from DE to VRAM address indicated by HL
;*******
ldrvrm:			xor	a
			ld	(rainbow_on),a		;Disable rainbow
			push	bc
			ld	a,(VDPportWRITE)	;Get VDP write port
			ld	c,a			;Keep it in C
			ld	a,l			;Low part of address
			di				;Disable interrupts
                	out     (c),a			;Select low part of VRAM address to write to
                	ld	a,h
			or	64
			ei				;Enable interrupts
			out     (c),a			;Select high part of VRAM address to write to
                	pop	bc
                	
			ex	de,hl			;Start address in HL
			ld	d,b
			ld	b,c			;Number of bytes in DB
			ld	a,(VDPportWRITE)	;Get VDP write port again
			dec	a
			ld	c,a			;Data port in register C
			ld	a,b
			or	a			;If low part ain't 0...
			jr	z,ldrvrmbc0
			inc	d			;...Increase high counter
ldrvrmbc0:		outi				;Output a byte
			jr	nz,ldrvrmbc0		;Inner loop
			dec	d
			jr	nz,ldrvrmbc0		;Outer loop
			ld	a,1
			ld	(rainbow_on),a		;Enable rainbow
			ret

;***********
;pchar_left: Print a character inclined to the left
;***********
pchar_left:		ex	de,hl			;Start address in HL
			ld	a,(VDPportWRITE)	;Get VDP write port
			dec	a
			ld	c,a			;Data port in register C

			ld	a,(hl)			;Get byte
			add	a,a			;Move one bit left
                	out     (c),a			;Output it
			inc	hl			;Next byte

			ld	a,(hl)			;Repeat the same 3 more times
			add	a,a
                	out     (c),a
			inc	hl
			ld	a,(hl)
			add	a,a
                	out     (c),a
			inc	hl
			ld	a,(hl)
			add	a,a
                	out     (c),a
			inc	hl
			jr	ldrvrmbyte3		;Keep outputting bytes, this time without inclination

;************
;pchar_right: Print a character inclined to the right
;************
pchar_right:		ex	de,hl			;Start address in HL
			ld	a,(VDPportWRITE)	;Get VDP write port
			dec	a
			ld	c,a			;Data port in register C

			ld	a,(hl)			;Get byte
			srl	a			;Move one bit right
                	out     (c),a			;Output it
			inc	hl			;Next byte
							;Repeat the same 3 more times
			ld	a,(hl)
			srl	a
                	out     (c),a
			inc	hl
			ld	a,(hl)
			srl	a
                	out     (c),a
			inc	hl
			ld	a,(hl)
			srl	a
                	out     (c),a
			inc	hl
			jr	ldrvrmbyte3		;Keep outputting bytes, this time without inclination

;*************
;putchar_down: Print a character, centered
;*************
putchar_down:		ex	de,hl			;Start address in HL
			ld	a,(VDPportWRITE)	;Get VDP write port
			dec	a
			ld	c,a			;Data port in register C
			ld	b,8			;Byte counter
			outi				;Output the byte in (HL) to the VDP
			nop
			nop				;Small pause for slow VPDs
							;Repeat for all bytes
			outi				
			nop
			nop
			outi
			nop
			nop
			outi
			nop
			nop
ldrvrmbyte3:		outi				;From here, 4 more characters
			nop
			nop
			outi
			nop
			nop
			outi
			nop
			nop
			outi
			jp	pmess_return		;Return to printmessage routine

;***********
;ldrvrmbyte: Copy a character from DE to VRAM address indicated by HL 
;***********
ldrvrmbyte:		ld	a,(VDPportWRITE)	;Get VDP write port
			ld	c,a			;Keep it in C
			ld	a,l			;Low part of address
			di				;Disable interrupts
                	out     (c),a			;Select low part of VRAM address to write to
                	ld	a,h
			or	64
			ei				;Enable interrupts
			out     (c),a			;Select high part of VRAM address to write to

			ex	de,hl			;HL points to character
			dec	c			;Data port
			ld	b,8			;Byte counter
			outi				;Output the byte in (HL) to the VDP
			nop
			nop				;Small pause for slow VPDs
							;Repeat for all bytes
			outi
			nop
			nop
			outi
			nop
			nop
			outi
			nop
			nop
			outi
			nop
			nop
			outi
			nop
			nop
			outi
			nop
			nop
			outi
			ret

;*********
;ldrvrm8k: Copy BC bytes of data from 0e000h to VRAM address indicated by HL
;*********
ldrvrm8k:		ld	de,0e000h
			push	bc
			ld	a,(VDPportWRITE)	;Get VDP write port
			
			ld	c,a			;Keep it in C
			ld	a,l			;Low part of address
                	out     (c),a			;Select low part of VRAM address to write to
                	ld	a,h
			or	64
			nop
			out     (c),a			;Select high part of VRAM address to write to
			ex	de,hl
			pop	bc
			ld	d,b
			ld	b,c			;Number of bytes in DB
			ld	a,(VDPportWRITE)	;Get VDP write port
			dec	a			;Data port instead of control
			ld	c,a			;Keep it in C
			ld	a,b
			or	a			;If low part ain't 0...
			jr	z,ldrvrmbc08k
			inc	d			;Increase high part
ldrvrmbc08k:		outi
			jr	nz,ldrvrmbc08k		;Inner loop
			dec	d
			jr	nz,ldrvrmbc08k		;Outer loop
			ret


;***********
;volcarmapa: Dump 9 tiles addresses from the map to the buffer of the camera pointed by IY
;***********
volcarmapa:		ld	c,3		;3 rows
			ld	a,(iy+CAM_MAPY)	;Get Y map position
			ld	e,0
			and	a		;Clear carry flag
			rra
			rr	e		;Rotate AE a bit to right
			rra
			rr	e		;Rotate AE another bit to right
			ld	d,a		;DE = MAPY*64
			ld	a,(iy+CAM_MAPX)
			add	a,e
			ld	e,a		;DE = (MAPY*64)+MAPX
			ld	hl,mapa
			add	hl,de		;HL = map position of upper left corner
			push	iy
			pop	ix		;IX = position in tilelist
vm_outerloop:		ld	b,3		;3 columns
			push	hl		;Keep position
vm_innerloop:		ld	a,(hl)		;Get value
			and	63		;Remove limit bits
			add	a,a		;Duplicate
			ld	e,a
			ld	d,tilelist/256
			ld	a,(de)		;Read first byte of tile address
			ld	(ix),a		;Write to camera buffer
			inc	ix
			inc	de
			ld	a,(de)		;Read second byte of tile address
			ld	(ix),a		;Write to camera buffer
			inc	ix
			ld	de,1		;Default increase for next column in the map
			bit	6,(hl)		;Check if we've reached the horizontal limit
			jr	z,vm_nowraphor
			ld	de,-63		;Wraparound increase for next column
vm_nowraphor:		add	hl,de		;Update map
			djnz	vm_innerloop	;Inner loop
			pop	hl		;Restore position
			ld	de,64		;Default increase for next row in the map
			bit	7,(hl)		;Check if we've reached the vertical limit
			jr	z,vm_nowrapver
			ld	de,-4032	;Wraparound increase for next row
vm_nowrapver:		add	hl,de		;Point to next row
			dec	c
			jr	nz,vm_outerloop	;Repeat for all rows
			ret

;********
;printbg: prints all 9 backgrounds tiles in a map half
;********
printbg:		push	iy
			pop	ix			;points to tiles buffer in the current map position
			ld	c,3			;up to 3 lines of 8x8 blocks to be printed
			ld	e,(iy+CAM_BUFFERPOS)
			ld	d,(iy+CAM_BUFFERPOS+1)	;DE = BUFFERPOS to write to
			ld	a,(iy+CAM_INTRAX)	;get initial tile horizontal position
			ld	(cezb_pos_x),a		;set for the sprite routine
			ld	a,(iy+CAM_INTRAY)	;get initial tile vertical position
pbgouterloop:		ld	(cezb_pos_y),a		;set for the sprite routine
			ld	b,3			;up to 3 columns of 8x8 blocks to be printed
pbginnerloop:		push	bc
			push	de
			call	printsprite_bg  	;print background tile
			pop	de
			pop	bc
			ld	hl,8 			;next buffer position 8 chars right
			add	hl,de
			ex	de,hl
			inc	ix
			inc	ix			;point to next map tile
			ld	a,(cezb_pos_x)
			add	a,8
			ld	(cezb_pos_x),a		;update X position for next block
			djnz	pbginnerloop		;loop for all 3 in a line
			sub	24			
			ld	(cezb_pos_x),a		;update X position for next block
			ld	hl,232			; = 256-24, 256 = next buffer position 8 chars down
			add	hl,de
			ex	de,hl
			ld	a,(cezb_pos_y)
			add	a,8			;update Y position for next block
			dec	c
			jr	nz,pbgouterloop		;repeat for 3 lines
			ret

;***************
;printsprite_bg: includes clipping, no transparencies (for backgrounds)
;***************
;DE=BUFFERPOS
;IY points to current camera
;IX points to the backgrounds buffer
printsprite_bg:		ld	l,(ix)		 	 ;19 | IX points to the backgrounds buffer
			ld	h,(ix+1)	 	 ;19 | HL = tile graph
			ld	a,(cezb_pos_y)		 ;10 |
			and	a			 ; 4 |
			jp	p,psbg_firstnoskipline 	 ;10 |  72

			;		always
			;total start > 	72

;first line, and until it stops skipping:
			ld	bc,8			;10  |
			add	hl,bc			;11  |	; was ADD HL, 24 (= 8 * 3), 1 line in tile. Now add hl, 8 (1 line in tile)
			ld	c,32			; 7  |
			ex	de,hl			; 4  |
			add	hl,bc			;11  |
			ex	de,hl			; 4  |	;was ADD DE, 48 (= 16 * 3), 1 line in buffer. Now add de,32 (1 line in buffer)
			ld	bc,1800			;10  |54  ; B=7,C=8

psbg_line5:		inc	a                       ; 4  |
			jr	z,psbg_firstnoskipline2 ;12,7|16

						;-5
			push	bc			;11  |
			ld	bc,32			;10  |
			ex	de,hl			; 4  |
			add	hl,bc			;11  |
			ex	de,hl			; 4  |	; was ADD DE, 48 (= 16 * 3), 1 line in buffer. Now add de,32 (1 line in buffer)
			ld	c,8			; 7  |
			add	hl,bc			;11  |	; was ADD HL, 24 (= 8 * 3), 1 line in tile. Now add hl, 8 (1 line in tile)
			pop	bc			;10  |
			djnz	psbg_line5		;13,8|76 ; update B

			;	        no skip / skip up
			;total line 1:  16      / 92

					; -5
			ret			;10		;return if we're above the buffer
			;total end:	 5

psbg_firstnoskipline:
			ld	a,d			; 4    | 
			cp	(iy+CAM_LIMITDOWN)	;19    |
			ret	nc			; 5,11 |16,22 ;Return if we're below the buffer
			ld	bc,2056			;10    | ; B=8,C=8

psbg_firstnoskipline2:
			ld	a,(cezb_pos_x)		;10  |
			and	a			; 4  |
			jp	m,psbg_left		;10  |39 ;if position is negative, go to cut characters in the left side
	
			cp	16			; 7  |
			ret	nc			;11,5|18 ;if position is above 16, return because it's at the right of the visible zone

						;-6
			sub	9			; 7  |
			jp	p,psbg_right		;10  |11 ;if the sprite doesn't fit, go to cut characters in the right side


			;		normal  / left   / right    / skip down / skip right 
			;total line2 > 	 86(70) / 67(39) /  98 (70) / 22        / 86(58)

psbg_line3:		ld	a,d			; 4  |
			cp	(iy+CAM_LIMITDOWN)	; 7  |
			ret	nc			;11,5|22 ;Return if we're below the buffer

						;-6  |
			push	bc			;11  |
			ld	b,c			; 4  |
			ld	c,d			; 4  |13 ;set C higher than 1*8 (we're sure D is), because ldi will decrease BC, and we don't want it touching B
			;		noskip / skip up
			;total line3 > 	35     / 22

psbg_char:		ldi				;16  |	 ;copy byte from background tile to buffer
			djnz	psbg_char		;13,8|61 ;repeat for all columns
							
			; 	       noskip
			;total char >  61

					; -5 |
			ld	c,24		; 7  |
			ex	de,hl		; 4  |
			add	hl,bc		;11  |
			ex	de,hl		; 4  |  ; point to next line in buffer
							; was ADD DE, 24 (= 8 * 3), 8 extra chars to jump a line (together with the 24 LDIs before it, makes 48 = 16*3)
							;now it's still DE, 24, 24 extra chars to jump a line (together with the 8 LDIs before it, makes 32)
			pop	bc		;10  |
			djnz	psbg_line3	;13,8|44

			;		no skip
			;total line4 >  44

					; -5
			ret			;10	;End
			;total end:	5

psbg_left:		inc	hl			; 6  |  ; increase tile position
			inc	de			; 6  |  ; increase buffer position
			dec	c			; 4  |	; decrease number of characters
			ret	z			;11,5| 49 ; return if we're completely at the left of buffer
		
						;-6  |
			inc	a			; 4  | ; increase position
			jr	nz,psbg_left		;12,7| 10 ;keep doing it until position is 0

						;-5  |
			;	      last / skip / final
			;total char >  54  /  59  /    49

psbg_lineleft:
			ld	a,d			; 4  |
			cp	(iy+CAM_LIMITDOWN)	; 7  |
			ret	nc			;11,5|22 ;Return if we're below the buffer

						;-6  |
			push	bc			;11  |   ;keep number of lines and characters
			push	hl			;11  |   ;graph position
			push	de			;11  |   ;buffer position
			ld	b,c			; 4  |   ;number of characters in B
			ld	c,d			; 7  |35 ;set C higher than 1*7 (we're sure D is), because ldi will decrease BC, and we don't want it touching B
			;		noskip / skip up
			;total line3 >  57     / 22

psbg_charleft:		ldi				;16  |   ;copy bytes from background tile to buffer
			djnz	psbg_charleft		;13,8|61 ;repeat for all visible columns

			; 	       noskip
			;total char >  61

					; -5  |
			pop	hl		;10   |	; restore buffer position
			ld	c,32		; 7   |
			add	hl,bc		;11   |
			ex	de,hl		; 4   | ; was ADD DE, 48, to advance 16*3 = 1 buffer line. Now it's ADD DE, 32 to advance a buffer line
			pop	hl		;10   | ; restore graph position
			ld	c,8		; 7   | ; was ADD HL, 24, to advance 8*3 = 1 tile line, now it's ADD HL, 8, to advance a tile line
			add	hl,bc		;11   |
			pop	bc		;10   | ; Restore number of lines
			djnz	psbg_lineleft	;13,8 |78  ;Repeat for all lines

			;		no skip
			;total line2 >  78

					; -5
			ret			;10	;End
			;total end: 5

psbg_right:		cpl				; 4  | 
			add	a,c			; 4  |
			ld	c,a			; 4  |   ; get number of visible characters
			jp	psbg_lineleft		;10  |22 ; right side is printed the same way as left

;**********
;movetyres: move tyres of the car, changing the bit 0 in the frame variable
;**********
movetyres:		ld	a,(ix+SP_TYRES)
			xor	1
			ld	(ix+SP_TYRES),a
			ret

;**************
;printcarsleft: print left side cars
;**************
printcarsleft:		ld	ix,car1			;First car
			ld	b,4			;Number of cars
pc_loop:		bit	4,(ix+SP_STATE)
			jp	nz,pc_noprint		;If the car doesn't race, don't print it
			
			ld	a,(carbuffer)		;Init the cars buffer
			ld	(ix+SP_BUFSTART),a

			ld	a,(ix+SP_MAPX)
			inc	a
			sub	(iy+CAM_MAPX)
			and	63
			cp	4
			jr	nc,pc_noprint		;If horizontal position if far, don't print this car
			ld	e,a
			ld	a,(ix+SP_MAPY)
			inc	a
			sub	(iy+CAM_MAPY)
			and	63
			cp	4
			jr	nc,pc_noprint		;If vertical position if far, don't print this car
			dec	a
			add	a,a
			add	a,a
			add	a,a
			add	a,(iy+CAM_INTRAY)
			add	a,(ix+SP_INTRAY)
			ld	(cezb_pos_y),a		;Store vertical position
			ld	d,a
			ld	a,e
			dec	a
			add	a,a
			add	a,a
			add	a,a
			add	a,(iy+CAM_INTRAX)
			add	a,(ix+SP_INTRAX)
			ld	(cezb_pos_x),a		;Store horizontal position
			ld	e,a
			ld	a,(iy+CAM_CAR)
			cp	(ix+SP_CAMERA)
			jr	nz,pc_nocamera
			ld	(iy+CAM_POSX),e
			ld	(iy+CAM_POSY),d		;If this is the main car pointed by the camera, store its position in the camera variables
pc_nocamera:		ld	a,e
			push	bc			;Keep BC for later
			ld	bc,0
			and	a
			jp	p,pc_nosignex
			dec	b			;B=255 if E is negative
pc_nosignex:		sra	d
			rr	c
			sra	d
			rr	c
			sra	d
			rr	c			;dc==y*32, sign extended
			add	a,c			;add c to e
			ld	e,a
			ld	a,d
			adc	a,b			;extend carry to d
			ld	d,a			;de=y*32+x
			ld	l,(iy+CAM_BUFFER)
			ld	h,(iy+CAM_BUFFER+1)
			add	hl,de			;hl=CAM_BUFFER+(y*16+x)
			ex	de,hl
			call	preprintsprite		;Print the car
			pop	bc
pc_noprint:		ld	a,(carbuffer)		;Check the buffer
			ld	e,a
			sub	(ix+SP_BUFSTART)	;Substract buffer start to current position
			rrca
			rrca				;Divide by 4
			ld	(ix+SP_BUFNUM),a	;Store number of chars in SP_BUFNUM
			ld	a,e
			ld	(ix+SP_BUFSTART),a	;Now SP_BUFSTART points to the last position
			ld	de,car2-car1
			add	ix,de			;Point to next car
			dec	b
			jp	nz,pc_loop		;Loop for all cars
			ret

;***************
;printcarsright: print right side cars
;***************
printcarsright:		ld	ix,car1			;First car
			ld	b,4			;Number of cars
pc_loopright:		bit	4,(ix+SP_STATE)
			jp	nz,pc_noprintright	;If the car doesn't race, don't print it

			ld	a,(ix+SP_MAPX)
			inc	a
			sub	(iy+CAM_MAPX)
			and	63
			cp	4
			jp	nc,pc_noprintright	;If horizontal position if far, don't print this car
			ld	e,a			;E = horizontal difference

			ld	a,(ix+SP_MAPY)
			inc	a
			sub	(iy+CAM_MAPY)
			and	63
			cp	4
			jp	nc,pc_noprintright	;If vertical position if far, don't print this car
			push	bc			;Keep number of cars in stack
			dec	a			;Decrease difference
			add	a,a
			add	a,a
			add	a,a			;*8
			add	a,(iy+CAM_INTRAY)	;Add camera's INTRAY
			add	a,(ix+SP_INTRAY)	;Add car's INTRAY to get vertical position
			ld	(cezb_pos_y),a		;Store vertical position
			ld	d,a			;Keep it in D
			ld	a,(carbuffer)		;Get carbuffer before printing
			push	af			;Keep it in stack
			ld	a,e			;Get horizontal difference again
			dec	a			;Decrease difference
			add	a,a
			add	a,a
			add	a,a			;Multiply by 8
			add	a,(iy+CAM_INTRAX)	;Add camera's INTRAX
			add	a,(ix+SP_INTRAX)	;Add car's INTRAX
			ld	(cezb_pos_x),a		;Store horizontal position
			ld	e,a			;Keep it in E
			ld	a,(iy+CAM_CAR)
			cp	(ix+SP_CAMERA)		;Compare car's ID with camera's
			jr	nz,pc_nocameraright
			ld	(iy+CAM_POSX),e
			ld	(iy+CAM_POSY),d		;If this is the main car pointed by the camera, store its position in the camera variables
pc_nocameraright:	ld	a,e			;Get horizontal position
;			push	bc
			ld	bc,0
			and	a
			jp	p,pc_nosignexright
			dec	b			;B=255 if E is negative
pc_nosignexright:	sra	d
			rr	c
			sra	d
			rr	c
			sra	d
			rr	c			;dc==y*32, sign extended
			add	a,c			;add c to e
			ld	e,a
			ld	a,d
			adc	a,b			;extend carry to d
			ld	d,a			;de=y*32+x
			ld	l,(iy+CAM_BUFFER)
			ld	h,(iy+CAM_BUFFER+1)
			add	hl,de			;hl=CAM_BUFFER+(y*16+x)
			ex	de,hl
			call	preprintsprite		;Print the car
;			pop	bc
			ld	a,(carbuffer)		;Check the buffer
			pop	hl
			sub	h
			jr	z,pc_noprintright2	;If no advance, the car hasn't entered the buffer

			rrca
			rrca				;Divide by 4
			ld	b,a			;B = number of right chars
			ld	h,buffer/256
			ld	d,h
			ld	a,(carbuffer)		;Current carbuffer position (low byte)
			dec	a			;Decrease 8 bit
			ld	e,a
			inc	de			;Increase 16 bits, just in case we were in the last position
							;Now DE points to left cars buffer
pcpcr_outbuc:		dec	e
			dec	e
			dec	e
			dec	e			;Point to a right char
			ld	l,(ix+SP_BUFSTART)	;HL = End of buffer for left side
			ld	a,(ix+SP_BUFNUM)	;Number of characters in the left side buffer for this car
			and	a
			jr	z,pc_noprintright2	;No char coincidence if car isn't printed in the left side
			ld	c,a			;C = number of left chars
			ld	a,(de)			;Get char code
pcpcr_innbuc:		dec	l
			dec	l
			dec	l
			dec	l			;Point to a left char
			cp	(hl)			;Check if there's a coincidence
			jr	z,pcpcr_gotcha
			dec	c
			jr	nz,pcpcr_innbuc		;Check for all left chars
pcprc_processed:	dec	b
			jr	nz,pcpcr_outbuc		;Check for all right chars
							;No coincidence here
pc_noprintright2:	pop	bc
pc_noprintright:	ld	de,car2-car1
			add	ix,de			;Point to next car
			dec	b
			jp	nz,pc_loopright		;Loop for all cars
			ret

pcpcr_gotcha:		inc	e			
			ld	a,(de)			;Get next byte to get third
			inc	l
			cp	(hl)			;Compare it too
			jr	nz,pcpcr_nosame
			sub	64			;Mark same third
pcpcr_nosame:		sub	128			;Mark repeated char
			ld	(de),a			;Update third
			dec	l			;Back to start of character
			ld	(ix+SP_BUFSTART),l	;Start looking next chars from here (same order)
			inc	e
			inc	e			;Go to last byte of right character
			ld	a,l
			ld	(de),a			;Mark offset from left
			dec	e
			dec	e
			dec	e			;Back to the start of right character
			dec	c
			ld	(ix+SP_BUFNUM),c	;Right characters left to test
			jr	pcprc_processed		;Continue testing

;***************
;preprintsprite: print sprite temporally, keeping it in tables until it's really printed to the screen buffer
;***************
;IX > points to current car
;IY > points to current camera
;DE > points to current BUFFERPOS
preprintsprite:		ld	l,(ix+SP_GRAPH)		 ;19 |
			ld	h,(ix+SP_GRAPH+1)	 ;19 |
			ld	a,(cezb_pos_y)		 ;10 |
			and	a			 ; 4 |
			jp	p,ps_firstnoskipline 	 ;10 |62

			;		always
			;total start > 	62

;first skipped line:
			ld	bc,3			;10 |
			add	hl,bc			;11 |	;was ADD HL, 9 (= 3 * 3), 1 line in the sprite. Now add hl, 3 (1 line in sprite)
			ld	c,32			; 7 |
			ex	de,hl			; 4 |
			add	hl,bc			;11 |   ;was ADD DE, 48 (= 16 * 3), 1 line in buffer. Now add de,32 (1 line in buffer)
			inc	a			; 4 |
			jp	z,ps_skipped1		;10 | 57
;second skipped line:
			add	hl,bc			;11 |
			ex	de,hl			; 4 |
			ld	c,3			; 7 |
			add	hl,bc			;11 |	;was ADD HL, 9 (= 3 * 3), 1 line in the sprite. Now add hl, 3 (1 line in sprite)
			ld	bc,259			;10 | 1*3
			inc	a			; 4 |
			jp	z,ps_skipped		;10 | 57
			ret				;10 | 10

ps_skipped1:		ex	de,hl			; 4  |
			ld	bc,515			;10  |	2 * 3
			jp	ps_skipped		;10  | 24

ps_firstnoskipline:	ld	a,d			; 4    |
			cp	(iy+CAM_LIMITDOWN)	; 7    |
			ret	nc			; 5,11 |16, 22
			ld	bc,771			;10    | size_y 3*3

		;		 	no skip / skip 1 / skip 2 / cut up / cut down
		;total skiplines > 	26	/     81 /    114 /    124 /       22

ps_skipped:		ld	a,(cezb_pos_x)		;10  |
			and	a			; 4  |
			jp	m,ps_left		;10  |24

			cp	16			; 7  |
			ret	nc			;5,11|12,18
			sub	14			; 7  |
			jp	p,ps_right		;10  |17

			;	       normal / left / right / skip right 
			;total line2 > 53     / 24   / 53    / 42

ps_line3:		ld	a,d			; 4  |
			cp	(iy+CAM_LIMITDOWN)	; 7  |
			ret	nc			;5,11|16,22

			push	bc			;11  |
			ld	b,c			; 4  |
			ld	c,d			; 4  |19

			;		noskip / skip down
			;total line3 > 	35     / 22

ps_char:		ld	a,(hl)			; 7  |		Get character
			and	a			; 4  |
			jr	nz,ps_solid		;12,7| 12
						;-5			Black character, do nothing
			inc	hl			; 6 |
			inc	de			; 6 |		Increase position
			jp	ps_fromgood		;10 | 39	Keep processing

ps_solid:		push	hl
			ld	hl,(carbuffer)		;Get carbuffer address
			cp	34
			jr	c,ps_notyres		;Below 34, there are no tyres
			add	a,(ix+SP_TYRES)		;Add tyres state
ps_notyres:		ld	(hl),a			;Store character in buffer
			inc	l
			ld	(hl),d			;Then store high byte of position
			inc	l
			ld	(hl),e			;Then store low byte of position
			inc	l
			ld	a,(ix+SP_COLOR)
			ld	(hl),a			;And finally store color
			inc	hl
			ld	(carbuffer),hl		;Update carbuffer
			pop	hl
			inc	de			; 6 |
			inc	hl			; 6 | 		Increase positions
ps_fromgood:		djnz	ps_char			;13,8 | 13	Keep processing

			; 	       normal / semitrans / skip blank
			;total char >     116 /       125 /         87

						;-5  |
			ld	c,29		; 7  |
			ex	de,hl		; 4  |
			add	hl,bc		;11  |	; was ADD DE, 39 ( == (16-3) * 3), now it's ADD DE, 29 (== (32-3) )
			ex	de,hl		; 4  |
			pop	bc		;10  |
			djnz	ps_line3	;13,8|

			;		no skip
			;total line4 >  44

					; -5
			ret			;10
			;total end:	5

ps_left:		inc	hl			; 6  |
			inc	de			; 6  |
			dec	c			; 4  |
			ret	z			;11,5|47
						;-6
			inc	a			; 4  |
			jr	nz,ps_left		;12,7|10

						;-5 |

	; 	     1 skip / 2 skip / 3 skip
	;total start >  52  /    109 /    161

ps_lineleft:		ld	a,d			; 4  |
			cp	(iy+CAM_LIMITDOWN)	; 7  |
			ret	nc			;11,5|22

						;-6
			push	bc			;11  |   lines & chars left
			push	hl			;11  |   graph position
			push	de			;11  |   buffer position
			ld	b,c			; 4  |
			ld	c,d			; 4  |35
			;		noskip / skip up
			;total line3 > 	57     / 22

ps_charleft:		ld	a,(hl)			; 7 |
			and	a			; 4 |
			jr	nz,ps_solidleft		;12,7| 12
							;-5 |
			inc	hl			; 6 |
			inc	de			; 6 |
			jp	ps_nextcharleft		;10 | 41

ps_solidleft:		push	hl
			ld	hl,(carbuffer)
			cp	34
			jr	c,ps_notyresleft
			add	a,(ix+SP_TYRES)
ps_notyresleft:		ld	(hl),a
			inc	l
			ld	(hl),d
			inc	l
			ld	(hl),e
			inc	l
			ld	a,(ix+SP_COLOR)
			ld	(hl),a
			inc	hl
			ld	(carbuffer),hl
			pop	hl
			inc	de			; 6 |
			inc	hl			; 6 |
ps_nextcharleft:
			djnz	ps_charleft		;13,8|13

			; 	       noskip normal / noskip semitrans / skip blank
			;total char >  116           / 125              / 87

					; -5
			pop	hl		;10   |
			ld	c,32		; 7   |
			add	hl,bc		;11   |	; was ADD DE, 48 (=16*3), now , 32
			ex	de,hl		; 4   |
			pop	hl		;11   |
			ld	c,3		; 7   |
			add	hl,bc		;11   |	;was ADD BC, 9 (=3*3), now 3
			pop	bc		;11   |
			djnz	ps_lineleft	;13,8 | 85

			;		no skip
			;total line4 >  85

						; -5 |
			ret			;10  | 5
			;total end: 5
ps_right:		cpl			; 4  |
			add	a,c		; 4  |
			ld	c,a		; 4  | 4    chars left in register C
			jp	ps_lineleft	;10  |18
			
;*********
;mixchars: Mix characters with background, if there's a need			
;*********
mixc_simple:		ex	af,af'		;recover car character
			ld	h,0
			ld	l,a
			add	hl,hl
			add	hl,hl
			add	hl,hl		;HL = A*8
			ld	de,car_tiles
			add	hl,de		;HL = car character
			push	iy		;12
			pop	de		;11	DE = IY
			ld	d,iyh
			ld	e,iyl
			ldi			;18
			ldi			;18
			ldi			;18
			ldi			;18
			ldi			;18
			ldi			;18
			ldi			;18
			ldi			;18	copy character
			ld	iyh,d
			ld	iyl,e
			push	de		;12
			pop	iy		;11	IY = DE
			jp	mixc_end
mixchars:		ld	iy,(carbuffer)	;IY = carbuffer current address
			xor	a
			ld	(iy+0),a	;Signal the end
			inc	iy		;Start saving data at carbuffer+1
			ld	(carbuffer),iy	;New carbuffer address
			ld	ix,buffer	;IX = buffer
mixc_nextchar:		ld	a,(ix+0)	; A = car character
			and	a
			ret	z		;Return if we reached the last one
			ex	af,af'		;keep it in A'
			ld	h,(ix+1)	;Get high part of screen position
			bit	7,h
			jp	z,mixc_repeated	;If bit 7 is 0, it's a repeated char

			ld	l,(ix+2)	;HL = screen position
			ld	a,(hl)		; A = background character
			cp	42
			jr	z,mixc_simple	;If background is black floor, just copy
			add	a,a		;tile under 128, *2
			ld	l,a
			ld	h,0		
			add	hl,hl
			add	hl,hl		;HL = background tile *8
			ld	de,tiles
			add	hl,de		;HL = background tile character
			exx			;HL' = background tile character
			ex	af,af'		;recover car character
			ld	h,0
			ld	l,a
			add	hl,hl		;car character*2
			ld	de,car_colors
			ex	de,hl
			add	hl,de
			ex	de,hl		;DE = car_color indications
			add	hl,hl
			add	hl,hl		;car character*8
			ld	bc,car_tiles
			add	hl,bc		;HL = car character address
mixc_nextsemi:		ld	a,(de)		;Get mix byte
			exx			;HL' = car, HL = background
			ld	b,4		;4 chars
			ld	c,a		;Mix byte in C
mixc_semichar:		rl	c		;Get 1 bit
			jr	nc,mixc_noc1	
			rl	c
mixc_c2:		exx			;HL = car, HL' = background
			ld	a,(hl)		;Get byte from car
			jr	mixc_putbyte
mixc_noc1:		rl	c
			jr	c,mixc_c2
			ld	a,(hl)		;Get byte from background
			exx			;HL = car, HL' = background
mixc_putbyte:		inc	hl
			exx			;HL' = car, HL = background
			inc	hl
			ld	(iy),a		;Store resulting byte
			inc	iy		;Point to next byte
			djnz	mixc_semichar	;Do for 4 half a character
			exx			;HL = car, HL' = background
			inc	de		;Point to next mix byte in the table
			ld	a,e
			and	1		;Check bit 0 of address
			jr	nz,mixc_nextsemi;If it's 1, we need to do the second round
			exx			;HL' = car, HL = background
mixc_end:		inc	ix
			inc	ix
			inc	ix
			inc	ix			;Point to next char in buffer
			jp	mixc_nextchar

mixc_repeated:		bit	6,h			;Check if it's in the same third
			jr	z,mixc_end		;If it is, no need to do anything
			push	hl
			push	de
			ld	a,(ix+3)		;numchar*4
			add	a,a			;*8
			ld	e,a
			ld	d,0
			ld	hl,(carbuffer)
			add	hl,de			;HL=origin
			ld	d,iyh
			ld	e,iyl			;DE = IY
			ldi
			ldi
			ldi
			ldi
			ldi
			ldi
			ldi
			ldi				;Copy char
			ld	iyh,d
			ld	iyl,e			;IY = DE
			pop	de
			pop	hl
			jr	mixc_end

;**********
;mixcolors: Mix colors with background, if there's a need to.	
;**********
mixcolors:		ld	(colorbuffer),iy	;Store colorbuffer start
			ld	ix,buffer		;IX = buffer
mixco_nextchar:		ld	a,(ix+0)		;A = car character
			and	a
			ret	z			;Return if we reached the last one
			ld	h,(ix+1)		;Get high part of screen position
			bit	7,h
			jp	z,mixco_repeated	;If bit 7 is 0, it's a repeated char

			ld	l,(ix+2)		;HL = screen position
			ld	l,(hl)			; L = background tile
			ld	h,0
			add	hl,hl
			add	hl,hl
			add	hl,hl			;HL = background tile * 8
			ld	de,colors
			add	hl,de			;HL = background tile color
			push	hl			;Keep in stack
			ld	h,0
			ld	l,a
			add	hl,hl			;HL = car character*2
			ld	de,car_colors
			ex	de,hl
			add	hl,de
			ex	de,hl			;DE = car_color indications
			pop	hl			;HL = tile color

mixco_nextsemi:		ld	a,(de)			;Get mix byte
			ld	b,4			;4 chars
			ld	c,a			;Mix byte in C
mixco_semichar:		rl	c			;Get 1 bit
			jr	nc,mixco_nomix		;0: no mix

			ld	a,(hl)			;1: mix, get background byte
			and	15			;background color from tile
			rl	c			;get another bit
			jr	nc,mixco_mixgray
			or	(ix+3)			;mix car color
			jr	mixco_putbyte
mixco_mixgray:		or	224			;mix gray from tyre
mixco_putbyte:		inc	hl			;next background position
			ld	(iy),a			;keep in buffer
			inc	iy			;next buffer position
			djnz	mixco_semichar		;repeat for the 4 chars
			inc	de			;next mix byte position
			ld	a,e
			and	1			;check bit 0 of that position
			jr	nz,mixco_nextsemi	;If it's 1, we need to do the second round
mixco_end:		inc	ix
			inc	ix
			inc	ix
			inc	ix			;Point to next char in buffer
			jp	mixco_nextchar
mixco_nomix:		rl	c			;get 1 bit
			jr	nc,mixco_background	;0: color from background
			ld	a,(ix+3)		;1: color from car
			jr	mixco_putbyte
mixco_background:	ld	a,(hl)			;get background color
			jr	mixco_putbyte

mixco_repeated:		bit	6,h			;Check if it's in the same third
			jr	z,mixco_end		;If it is, no need to do anything
			push	hl
			push	de
			ld	a,(ix+3)		;numchar*4
			add	a,a			;*8
			ld	e,a
			ld	d,0
			ld	hl,(colorbuffer)
			add	hl,de			;HL=origin
			ld	d,iyh
			ld	e,iyl			;DE = IY
			ldi
			ldi
			ldi
			ldi
			ldi
			ldi
			ldi
			ldi				;Copy char
			ld	iyh,d
			ld	iyl,e			;IY = DE
			ld	a,(ix+1)
			or	0c0h			
			ld	(ix+1),a		;restore position for last step
			pop	de
			pop	hl
			jr	mixco_end

;************
;assignchars: Assign character number to the previously mixed car characters
;************
assignchars:		ld	hl,(carbuffer)		;HL' = carbuffer, to be written with char data
			ld	de,(colorbuffer)	;DE' = colorbuffer, to be written with color data
			exx
			ld	a,(shadowchar1)
			xor	86			;Switch between 182<->224
			ld	(shadowchar1),a
			ld	c,a			;First third start char
			ld	a,(shadowchar2)
			xor	64			;Switch between 128<->192
			ld	(shadowchar2),a
			ld	d,a			;Second third start char
			ld	e,224			;Fixed third third start char
			
			ld	ix,buffer		;this is the buffer where the characters are referenced with 4 bytes (character,position high, position low, color)
			
			ld	hl,(carbuffer)		;carbuffer, to read char data
			ld	iy,(colorbuffer)	;colorbuffer, to read color data

assic_nextchar:		ld	a,(ix)			;A = car character
			and	a
			ret	z			;Return if we reached the last one
			ld	a,(ix+1)		;Get high part of screen position
			and	a
			jp	p,assic_repeatedchar	;If bit 7 is 0, it's a repeated char
			cp	(thebigbuffer/256)+1	;check the third
			jr	z,assic_2nd
			jr	c,assic_1st
						;3rd third: always copy characters to be defined after the big screen dump
			ld	a,e			;a = char number
			inc	e			;increase 3rd third character number
			exx
			ld	b,(ix+1)
			ld	c,(ix+2)
			ld	(bc),a			;Print char in screen buffer
			exx
assic_copychar1st:	ld	b,8			;number of bytes
assic_copychar:		ld	a,(hl)			;get byte from buffer
			inc	hl
			exx
			ld	(hl),a			;write byte in carbuffer
			inc	hl
			ld	a,(iy)			;get byte from colorbuffer
			inc	iy
			ld	(de),a			;write in colorbuffer
			inc	de
			exx
			djnz	assic_copychar
assic_tonextchar:	inc	ix
			inc	ix
			inc	ix
			inc	ix			;point to next char in the buffer
			jr	assic_nextchar
			
						;1st third: the first 8 characters will be defined in VRAM, the rest copied to be defined before the big screen dump
assic_1st:		ld	a,c			;A = char number
			exx
			ld	b,(ix+1)
			ld	c,(ix+2)
			ld	(bc),a			;Print char in screen buffer
			exx
			inc	c			;increase char number
			cp	232
			jr	nc,assic_copychar1st	;Above 231: copy character in table
							;Below 232, define the character in VRAM
			push	bc
			push	de
			push	hl
			ld	d,h
			ld	e,l			;DE = RAM origin > next char in buffer
			ld	h,0
			ld	l,a
			add	hl,hl
			add	hl,hl
			add	hl,hl			;HL = VRAM destination A*8, 1st third char
			push	hl
			call	ldrvrmbyte		;Define character
			ld	d,iyh
			ld	e,iyl			;DE = RAM origin -> next color in buffer
			pop	hl
			ld	a,h
			or	20h
			ld	h,a			;HL = VRAM destination A*8+2000h, 1st third color
			call	ldrvrmbyte		;Define colors
			ld	bc,8
			add	iy,bc			;Update IY
			pop	hl
			add	hl,bc			;Update HL
			pop	de
			pop	bc
			ld	a,c
			cp	190			;Reached the jump point?
			jr	nz,assic_tonextchar
			ld	c,232			;Jump here
assic_tonextchar2:	jr	assic_tonextchar

						;2nd third: always define the new car tiles in VRAM
assic_2nd:		ld	a,d			;A = char number
			inc	d			;increase char number
			exx
			ld	b,(ix+1)
			ld	c,(ix+2)
			ld	(bc),a			;Print char in screen buffer
			exx
			push	bc
			push	de
			push	hl
			ld	d,h
			ld	e,l			;DE = RAM origin > next char in buffer
			ld	h,0
			ld	l,a
			add	hl,hl
			add	hl,hl
			add	hl,hl			;HL = char * 8
			ld	bc,800h
			add	hl,bc			;HL = VRAM destination A*8+800h, 2nd third char
			push	hl
			call	ldrvrmbyte		;Define character
			ld	d,iyh
			ld	e,iyl			;DE = IY
			pop	hl
			ld	a,h
			or	20h			;Add color bit
			ld	h,a			;HL = VRAM destination A*8+2800h, 2nd third color
			call	ldrvrmbyte		;Define colors
			ld	bc,8
			add	iy,bc			;Update IY
			pop	hl
			add	hl,bc			;Update HL
assic_2ndfromrepeat:	pop	de			;Restore counters
			pop	bc			;Restore counters
			jr	assic_tonextchar2

							;repeated chars can only belong to 2nd third, the 3rd third ones were stripped of their
							;	"repeated" status after copying their mixed color information
assic_repeatedchar:     push	de
			push	hl
			ld	h,buffer/256		;Copy the character
			ld	l,(ix+3)		;Get repeated char
			ld	d,0c1h			;Fixed 2nd third
			inc	l
			inc	l
			ld	e,(hl)			;Get address
			ld	a,(de)			;Get assigned char for this one
			exx
			ld	b,0c1h			;Second third
			ld	c,(ix+2)		;Low part of address
			ld	(bc),a			;Print char in screen buffer
			exx
			pop	hl
			pop	de
			jr	assic_tonextchar2

;***************
;preparesprites: Set sprites for the tyres
;***************
preparesprites:		ld	ix,buffer		;this is the buffer where the characters are referenced with 4 bytes (character,position high, position low, color)
			ld	iy,spritesbuffer+16	;Sprites buffer start for tyres (the first 4 sprites are for the car pixels in the map)
preps_next:		ld	a,(ix+0)		;A = car character
			ld	c,a			;Keep it in C
			and	a			
			ret	z			;exit when we reach the last
			jp	p,preps_nosprite	;characters below 128 have no sprites
			and	56
			jr	nz,prep_nodoublesized	;If bits 5-3 aren't 0, the sprite ain't double sized
			ld	a,223
			cp	(ix+2)			;Check if we're in the last line
			jr	nc,prep_nosecondhalf
			ld	a,c
			add	a,32			;Print only the first part
			ld	c,a
			jr	prep_nosecondhalf2

prep_nodoublesized:	xor	48			;Check if it's a second half
			jr	nz,prep_nosecondhalf
			ld	a,(ix+2)
			cp	32
			jr	nc,preps_nosprite	;Only print it if we're in the first line
prep_nosecondhalf:	ld	a,c			;Get the sprite
prep_nosecondhalf2:	srl	a		
			ld	e,a			;Divide by 2
			ld	d,0
			ld	hl,spriteheight-64	;Heights base -(128/2)
			add	hl,de
			ld	b,(hl)			;B = low height part
			ld	h,(ix+1)
			ld	l,(ix+2)		;HL = screen position
			ld	a,l
			and	31			;Keep row
			add	hl,hl
			add	hl,hl
			add	hl,hl			;Multiply by 8
			ld	(iy+1),l		;Horizontal position
			ld	a,h
			add	a,a
			add	a,a
			add	a,a
			add	a,b			;Adjust height
;			dec	a
			ld	(iy+0),a		;Vertical position
			ld	a,14			;Gray color
			ld	(iy+3),a		;Set color
			ld	a,c
			and	127
			ld	(iy+2),a		;Sprite form number
			inc	iy
			inc	iy
			inc	iy
			inc	iy			;Point to next sprite
preps_nosprite:		inc	ix
			inc	ix
			inc	ix
			inc	ix			;Point to next character in buffer
			jp	preps_next

;**********
;crearmapa: Create map, prepare all cars to start race
;**********
crearmapa:		ld	de,mapa
			call	deexo			;Uncompress map to mapa buffer
			ld	hl,mapa-1
			ld	de,64			;Lenght of a row
			ld	b,e			;Number of rows
cm_rightlimit:		add	hl,de
			ld	a,(hl)
			or	64
			ld	(hl),a			;Mark right limit
			djnz	cm_rightlimit		
			ld	b,64
cm_downlimit:		ld	a,(hl)
			or	128
			ld	(hl),a			;Mark low limit
			dec	l
			djnz	cm_downlimit
			ld	hl,mapa_checks
			ld	de,mapa_checks+256
			ld	bc,256
			ldir				;Copy the checkpoints
			dec	d
			ld	a,(de)
			rrca				;Divide by 2
			sub	130			;Adjust number
			ld	b,a			;Number of pair of values to enlarge
;			ld	l,1
cm_novicecheck:		inc	l
			ld	a,(hl)
			dec	a			;Decrease left/up limit
			jp	m,cm_nv_noleft		;If it becomes negative, don't touch it
			ld	(hl),a
cm_nv_noleft:		inc	l
			ld	a,(hl)
			inc	a			;Increase right/down limit
			cp	64
			jr	z,cm_nv_nodown		;If it becomes 64, don't touch it
			ld	(hl),a
cm_nv_nodown:		djnz	cm_novicecheck

			ld	a,(recording)
			cp	2			;Check if we're replaying
			jr	nz,cm_nv_randomize
			ld	a,(startpositions_rec)	;Yes: Get the same random value that we got when we played
			jr	cm_nv_norand
			
cm_nv_randomize:	ld	a,r			;No: Get random value
			and	31			;Keep the 5 lower bits
			ld	(startpositions_rec),a	;Store value for later replay
cm_nv_norand:		ld	h,startpositions/256
			ld	l,a
			ld	a,(hl)			;Take random start positions
			ld	l,a
			ld	a,(de)			;Get start of positions
			ld	h,a
			ld	ix,car1			;First car
			ld	b,4			;Number of cars
cm_set_car:		push	bc
			ld	e,h			;E = start of positions
			rr	l			
			jr	c,cm_nojump1
			inc	e			;Increase position if first bit is 0
cm_nojump1:		rr	l
			jr	c,cm_nojump2
			inc	e			;Increase position twice if second bit is 0
			inc	e			
cm_nojump2:		push	hl
			ld	a,(de)			;Get mapx for this car
			ld	(ix+SP_MAPX),a		;Set it
			ld	a,4
			add	a,e
			ld	e,a			;Point to next data
			ld	a,(de)			;Get mapy for this car
			ld	(ix+SP_MAPY),a		;Set it
			ld	a,4
			add	a,e
			ld	e,a			;Point to next data
			ld	a,(de)			;Get intrax for this car
			ld	(ix+SP_INTRAX),a	;Set it
			ld	a,4
			add	a,e
			ld	e,a			;Point to next data
			ld	a,(de)			;Get intray for this car
			ld	(ix+SP_INTRAY),a	;Set it
			ld	a,4
			add	a,e
			ld	e,a			;Point to next data
			ld	a,(de)			;Get frame for this car
			ld	(ix+SP_FRAME),a		;Set it
			rra				;Divide by 2
			inc	a			;Increase
			ld	hl,CarGraph-9		;Graphic address - 1 frame
			ld	bc,9			;Each frame has 3x3 bytes
cm_set_frame:		add	hl,bc
			dec	a
			jr	nz,cm_set_frame		;Get graphic address in HL
			xor	a
                	ld      (ix+SP_LASTCONTROL),c	;BUG: a 9 could be a valid value.
			ld	(ix+SP_SUBX),a		;Reset a bunch of variables
			ld	(ix+SP_SUBY),a
			ld	(ix+SP_GRAPH),l
			ld	(ix+SP_GRAPH+1),h	;Set graphic address
			ld	(ix+SP_SPEED),a
			ld	(ix+SP_SUBSPEED),a
			ld	(ix+SP_SUBGIRO),a
			ld	(ix+SP_MINIMAPX),a
			ld	(ix+SP_MINIMAPY),a
			ld	(ix+SP_LAPS),a
			ld	(ix+SP_COL_DUR),a
			ld	(ix+SP_COL_DIR),a
			ld	(ix+SP_MAPPOINTS),a
			ld	(ix+SP_BESTLAP),a
			ld	(ix+SP_BESTLAP+1),a
			ld	(ix+SP_LAPSTART),a
			ld	(ix+SP_LAPSTART+1),a
			ld	(ix+SP_TIMESTAMP),a
			ld	(ix+SP_TIMESTAMP+1),a
			ld	(ix+SP_TIMESTAMPLAP),a
			ld	(ix+SP_TIMESTAMPCP),a
			ld	(ix+SP_TYRES),a
			inc	a
			ld	(ix+SP_NEXTCHECK),a	;First check at 1
			ld	a,(ix+SP_STATE)		;Get state
			and	144			;Preserve important bits (human & playing)
			ld	(ix+SP_STATE),a		;Update state
			ld	a,(ix+SP_CAMERA)
			ld	iy,camera_left
			cp	(iy+CAM_CAR)		;Car is pointed by the left camera?
			jr	z,dosetcamera
			ld	iy,camera_right
			cp	(iy+CAM_CAR)		;Car is pointed by the right camera?
			jr	nz,nosetcamera
dosetcamera:		call	setcamera		;If a car is pointed by a camera, set this camera too
nosetcamera:		push	de
			call	direction_update	;Update direction according to frame
			pop	de
			ld	bc,car2-car1
			add	ix,bc			;Point to next car
			pop	hl
			pop	bc
			dec	b
			jp	nz,cm_set_car		;Loop for all cars
			ret

;***************
;minimap_center: Prepare the minimap at the center of the screen (for interlude)
;***************
minimap_center:		ld	ix,screenbuffer+268	;12th character of second third, centered minimap
			jr	minimap_define_chars2

;*********************
;minimap_define_chars: Prepare the minimap at the low part of the screen (starting the stage)
;*********************
minimap_define_chars:	ld	ix,screenbuffer+520	;8th character of third third, where minimap starts
minimap_define_chars2:	ld	iyl,132			;First char
			ld	hl,mapa			;Map definition zone
			ld	de,buffer		;Where to store the characters
			ld	b,8			;8 rows
looplv3:		push	bc
			ld	b,8			;8 characters per row
looplv2:		push	bc
			ld	b,8			;8 lines per character
			push	hl
			push	de
			push	de
looplv1:		push	bc
			ld	c,1			;Marker bit
looplv0:		ld	a,(hl)			;Get map 
			and	63			;Discard two higher bits
			xor	51			;Grass value
			cp	1			;If there was grass, carry is set
			ccf				;Complement carry
			rl	c			;Enter into C
			inc	hl			;Next map address
			jr	nc,looplv0		;Bit processed
			ld	a,c
			ld	(de),a			;Store into buffer
			inc	de			;Next buffer position
			ld	bc,56
			add	hl,bc			;Now go for the next line of the same character in the map
			pop	bc
			djnz	looplv1			;Loop until char is processed
			pop	hl
			ld	b,8			;Check 8 bytes
			xor	a			;Start at 0
orit:			or	(hl)			;Or value
			inc	hl			;Next value
			djnz	orit			;Or all 8 values
			and	a			;0?
			jr	z,skipchar		;If all bytes were 0, skip this char.
setchar:		ld	a,iyl			;Assign this index
			inc	iyl			;Increase index
			pop	hl			;Discard register from stack
			jr	noskipchar
skipchar:		pop	de			;Restore DE
			ld	a,131			;Assign void map character
noskipchar:		ld	(ix+0),a		;Print assigned char in the screen
			inc	ix			;Next position
			pop	hl			;Restore HL
			ld	bc,8			
			add	hl,bc			;Next character in the map
			pop	bc
			djnz	looplv2			;Do for all 8 characters in a line
			ld	c,24
			add	ix,bc			;Point to next line in screen
			ld	bc,448
			add	hl,bc			;Point to next line in the map
			pop	bc
			djnz	looplv3			;Repeat for 8 lines
			ret

switchcar:		ld	de,(lastcamchange)	;Check last time a camera was changed
			ld	hl,(main_ticks)
			and	a			;No carry
			sbc	hl,de			;Compare current time to time of last camera change
			ld	a,h
			and	a
			jr	nz,switch_ok		;If difference is over 256, it's ok
			ld	a,l
			cp	8			;Difference over 7?
			ret	c			;No: Wait at least 0.8 secs to make another camera change
switch_ok:		add	hl,de			;Reverse last change
			ld	(lastcamchange),hl	;Update last sound change timestamp
			ld	ix,car1			;First car
			ld	a,(other)		;The car of the other camera
			ld	b,a
			ld	a,(iy+CAM_CAR)		;Get current car
switch_next:		inc	a			;Increase it
			and	3			;Keep it in 0-3 range
			cp	b			;Compare with the one from the other camera
			jr	z,switch_next		;If it's the same, try another
			ld	(iy+CAM_CAR),a		;Set new car in the camera
			and	a
			jr	z,setcamera
			ld	b,a			;If it's not the first car
			ld	de,car2-car1
switch_add:		add	ix,de			;Update IX with the proper car
			djnz	switch_add

;**********
;setcamera: Set the camera for a certain car
;**********
setcamera:		bit	4,(ix+SP_STATE)		;Check if the car is racing
			jr	nz,switchcar		;No: keep switching car
			push	de			;Keep DE
			ld	a,(ix+SP_CAMERA)
			ld	(iy+CAM_CAR),a		;Set car identificator
			ld	b,6
			ld	(iy+CAM_POSX),b		
			ld	(iy+CAM_POSY),b		;Initially set car in the centre of the camera (relative to upper left corner) -> 6,6
			ld	d,(ix+SP_MAPX)		;Get car MAPX position
			ld	a,(ix+SP_INTRAX)	;And INTRAX position
			sub	b			;Calculate new INTRAX for the camera
			jr	nc,stc_nocarryx
			dec	d			;If it became out of bounds, adjust MAPX 
			add	a,8			; and adjust INTRAX
stc_nocarryx:		ld	b,a
			ld	(iy+CAM_MAPX),d		;Store MAPX for the cam
			xor	a
			sub	b			;Camera's INTRAX is in -7 to 0 range
			ld	(iy+CAM_INTRAX),a	;Store INTRAX for the cam
			ld	de,-1			;WAS -3 , now -1
			ld	l,(iy+CAM_BUFFER)
			ld	h,(iy+CAM_BUFFER+1)	;Buffer position start
			and	a
			jr	z,stc_nosubstractx
stc_substractx:		add	hl,de			;Substract 1 until we've compensated all INTRAX
			djnz	stc_substractx
stc_nosubstractx:	ld	d,(ix+SP_MAPY)		;Now repeat the same for Y coordinate
			ld	a,(ix+SP_INTRAY)
			sub	6
			jr	nc,stc_nocarryy
			dec	d
			add	a,8
stc_nocarryy:		ld	b,a
			ld	(iy+CAM_MAPY),d
			xor	a
			sub	b
			ld	(iy+CAM_INTRAY),a
			ld	de,-32			;Substract 32 for vertical position adjust
			and	a
			jr	z,stc_nosubstracty
stc_substracty:		add	hl,de
			djnz	stc_substracty
stc_nosubstracty:	ld	(iy+CAM_BUFFERPOS),l
			ld	(iy+CAM_BUFFERPOS+1),h	;Store BUFFERPOS

			ld	b,6
stc_campan:		push	bc
			call	camerapanning		;Pan the camera to set the car to the desired position
			pop	bc
			djnz	stc_campan		;Do it 6 times, to make sure we reach the place

			push	iy
			call	decolor			;Get marker colors
			pop	iy
			ld	de,buffer		;Marker colors start
			ld	a,20h
			or	(iy+CAM_VELOC)		;Get velocimeter address for colors
			ld	h,a
			ld	l,9*8			;Offset
			
			ld	bc,8*23			;23 characters
			call	ldrvrm			;Copy color 000-022 to 201-223

			pop	de
			ret

;*****************
;direction_update: Update the direction our car is looking to, changing the base speed depending on current frame
;*****************
direction_update:	ld	a,(ix+SP_FRAME)
			srl	a
			ld	e,a
			ld	d,0		;DE = frame/2 (==direction)
			ld	hl,speeds
			add	hl,de		;HL points to vertical speed
			ld	a,(hl)
			ld	(ix+SP_DIRY),a	;Set vertical direction
			ld	a,l
			sub	4		;Horizontal is 4 bytes away in the table
			ld	l,a
			ld	a,(hl)
			ld	(ix+SP_DIRX),a	;Set horizontal direction
			ret

;**************
;camerapanning: Change the camera position so the car we are pointing to gets nearer to the desired position, depending on its direction.
;**************
camerapanning:		ld	a,(ix+SP_FRAME)
			and	254			;Clear bit 0 from the frame
			ld	c,a
			ld	b,0
			ld	hl,position_wanted
			add	hl,bc			;HL = points to position_wanted
			ld	a,(hl)			;Get horizontal desired position
			inc	hl
			push	hl
			cp	(iy+CAM_POSX)		;Compare to current position
			jr	z,cam_nohorizontal	;If zero, no need to change
			jr	c,cam_left		;If carry, move left
			call	scrollbg_left		;Scroll background left to move right
			inc	(iy+CAM_POSX)		;Increase cam position
cam_nohorizontal:	pop	hl
			ld	a,(hl)			;Get vertical desired position
			cp	(iy+CAM_POSY)		;Compare to current position
			jr	z,cam_novertical	;If zero, no need to change
			jr	c,cam_up		;If carry, move up
			call	scrollbg_up		;Scroll background up to move down
			inc	(iy+CAM_POSY)		;Increase cam position
cam_novertical: 	ret
cam_left:		call	scrollbg_right		;Scroll background right to move left
			dec	(iy+CAM_POSX)		;Decrease cam position
			jp	cam_nohorizontal
cam_up:			call	scrollbg_down		;Scroll background down to move up
			dec	(iy+CAM_POSY)		;Decrease cam position
			jp	cam_novertical

;**************
;scrollbg_down: Scroll the background up, so we're moving down
;**************
scrollbg_down:		ld	l,(iy+CAM_BUFFERPOS)
			ld	h,(iy+CAM_BUFFERPOS+1)	;Get bufferpos
			ld	de,-32			
			add	hl,de			;One line up
			ld	a,(iy+CAM_INTRAY)	;Get INTRAY
			dec	a			;Decrease it
			cp	-8			;Check if we've reached -8
			jr	nz,scrollbg_down_end
							;If we've reached -8, adjust MAPY
			ld	a,(iy+CAM_MAPY)		;Get MAPY
			inc	a			;Increase it
			and	63			;Keep it in 0-63 range
			ld	(iy+CAM_MAPY),a		;Update MAPY
			ld	de,256
			add	hl,de			;Adjust BUFFERPOS 8 lines down
			xor	a			;And reset INTRAY
scrollbg_down_end:	ld	(iy+CAM_INTRAY),a	;Update INTRAY
			ld	(iy+CAM_BUFFERPOS),l
			ld	(iy+CAM_BUFFERPOS+1),h	;Update BUFFERPOS
			ret

;************
;scrollbg_up: Scroll the background down, so we're moving up
;************
scrollbg_up:		ld	l,(iy+CAM_BUFFERPOS)
			ld	h,(iy+CAM_BUFFERPOS+1)	;Get bufferpos
			ld	de,32
			add	hl,de			;One line down
			ld	a,(iy+CAM_INTRAY)	;Get INTRAY
			inc	a			;Increase it
			cp	1			;Check if we've reached 1
			jr	nz,scrollbg_up_end
							;If we've reached 1, adjust MAPY
			ld	a,(iy+CAM_MAPY)		;Get MAPY
			dec	a			;Decrease it
			and	63			;Keep it in 0-63 range
			ld	(iy+CAM_MAPY),a		;Update MAPY
			ld	de,-256
			add	hl,de			;Adjust BUFFERPOS 8 lines up
			ld	a,-7			;And reset INTRAY to -7
scrollbg_up_end:	ld	(iy+CAM_INTRAY),a	;Update INTRAY
			ld	(iy+CAM_BUFFERPOS),l
			ld	(iy+CAM_BUFFERPOS+1),h	;Update BUFFERPOS
			ret

;***************
;scrollbg_right: Scroll the background left, so we're moving right
;***************
scrollbg_right:		ld	l,(iy+CAM_BUFFERPOS)
			ld	h,(iy+CAM_BUFFERPOS+1)	;Get bufferpos
			ld	de,-1
			add	hl,de			;One char left
			ld	a,(iy+CAM_INTRAX)	;Get INTRAX
			dec	a			;Decrease it
			cp	-8			;Check if we've reached -8
			jr	nz,scrollbg_right_end
							;If we've reached -8, adjust MAPX
			ld	a,(iy+CAM_MAPX)		;Get MAPX
			inc	a			;Increase it
			and	63			;Keep it in 0-63 range
			ld	(iy+CAM_MAPX),a		;Update MAPX
			ld	de,8
			add	hl,de			;Adjust BUFFERPOS 8 characters right
			xor	a			;And reset INTRAX
scrollbg_right_end:	ld	(iy+CAM_INTRAX),a	;Update INTRAX
			ld	(iy+CAM_BUFFERPOS),l
			ld	(iy+CAM_BUFFERPOS+1),h	;Update BUFFERPOS
			ret

;**************
;scrollbg_left: Scroll the background right, so we're moving left
;**************
scrollbg_left:		ld	l,(iy+CAM_BUFFERPOS)
			ld	h,(iy+CAM_BUFFERPOS+1)	;Get bufferpos
			ld	de,1			
			add	hl,de			;One char right
			ld	a,(iy+CAM_INTRAX)	;Get INTRAX
			inc	a			;Increase it
			cp	1			;Check if we've reached 1
			jr	nz,scrollbg_left_end
							;If we've reached 1, adjust MAPX
			ld	a,(iy+CAM_MAPX)		;Get MAPX
			dec	a			;Decrease it
			and	63			;Keep it in 0-63 range
			ld	(iy+CAM_MAPX),a		;Update MAPX
			ld	de,-8
			add	hl,de			;Adjust BUFFERPOS 8 characters left
			ld	a,-7			;And reset INTRAX to -7
scrollbg_left_end:	ld	(iy+CAM_INTRAX),a	;Update INTRAX
			ld	(iy+CAM_BUFFERPOS),l
			ld	(iy+CAM_BUFFERPOS+1),h	;Update BUFFERPOS
			ret

;************
;velocimetro: Display the speedometer, and set sound parameters if necessary
;************
velocimetro:		ld	de,velo_sinaguja
			ld	h,(iy+CAM_VELOC)	;Get the VRAM address for this camera
			ld	l,8*3
			ld	bc,8*6			;6 characters
			call	ldrvrm			;Restore speedometer without needle
			ld	d,velocidad/256		;Points to velocidad
			ld	h,(iy+CAM_VELOC)	;Get the VRAM address for this camera again
			ld	a,(current_sound)
			srl	a
			cp	(iy+CAM_ID)		;If the sounding car belongs to this camera
			jr	nz,veloc_nosetsnd
			ld	a,(ix+SP_STATE)		
			ld	(snd_state),a		;Set sound state
			ld	a,(ix+SP_SPEED)
			ld	(snd_speed),a		;Set sound speed
veloc_nosetsnd:		ld	a,(ix+SP_SPEED)		;Get current speed
			add	a,a
			add	a,a
			add	a,a
			add	a,a			;Multiply by 16
			ld	e,a			;DE = graphics to be included

			rlca
			rlca				;Move 2 higher bits to lowest positions
			and	3			;Keep only those two bits
			cp	2			;
			sbc	a,-4			;Add 4 or 5 depending on value

			add	a,a
			add	a,a
			add	a,a			;Multiply by 8

			ld	l,a			;HL = VRAM address to be updated.
			ld	bc,8*2			;2 characters to be printed
			jp	ldrvrm			;Print needle and return

;********
;decolor: Get marker colors for our car in the buffer
;********
decolor:		ld	hl,mcolorspacked		;Colors are packed here
			ld	de,buffer			;And will be unpacked here
			ld	iyh,128				;First character
			ld	c,(ix+SP_COLOR)			;Get car's main color
			ld	b,67				;First 8 characters and three more byte
decol_put:		ld	a,c				;Color here
decol_putcolor:		ld	(de),a				;Store color
			inc	de				;Next position
			djnz	decol_putcolor			;Loop for the 67 bytes
decol_mainloop:		call	exo_getbit			;Get a bit
			jr	nc,decol_normalcolor		;bit 0 means normal color
								;bit 1 means a special color
			ld	b,1				;1 byte only
			ld	c,(ix+SP_COLORMARKER_DOUBLE)	;Try both colors
			call	exo_getbit			;Get bit
			jr	nc,decol_put			;Bit 0 means both colors
								;Bit 1 means another thing
			call	exo_getbit			;Get bit
			ret	c				;Bit 1 means end it
								
			ld	bc,6*256+240			;Bit 0 means white color, 6 bytes long
			jr	decol_put			;Put it

decol_normalcolor:	ld	c,(ix+SP_COLOR)			;Get normal color
			call	exo_getbit			;Get a bit
			jr	nc,decol_getsize		;Bit 0 means normal color
			ld	c,(ix+SP_COLORMARKER_SEC)	;Bit 1 means secondary color
decol_getsize:		ld	b,32				;Marker bit
decol_gettingsize:	call	exo_getbit			;Get a bit
			rl	b				;Enter it into register B
			jr	nc,decol_gettingsize		;Until marker bit is gone (3 bits)
			jr	decol_put			;Keep setting colors

;**************
;get_attr_self: Get the attribute under your car
;**************
get_attr_self:		ld	e,1
			ld	l,e			;Both displazements are 1, to point the center of your car

;*********
;get_attr: Get the attribute at current vertical position+register E, current horizontal position+register L
;*********
get_attr:		ld	b,(ix+SP_MAPY)		;Get MAPY and INTRAY coordinates
			ld	a,(ix+SP_INTRAY)
			add	a,e			;Adjust position with displacement
			jp	m,ga_negy
ga_incy:		cp	8
			jr	c,ga_noincy
			inc	b			;IF INTRAY is over 8, adjust with MAPY until it's below 8
			sub	8
			jr	ga_incy
ga_negy:		dec	b			;If INTRAY becomes negative, adjust with MAPY until it's positive
			add	a,8
			jp	m,ga_negy
ga_noincy:		ld	(ga_intray),a		;Store INTRAY
			ld	a,b
			and	63			;Wraparound map if necessary
			ld	e,0
			rra
			rr	e
			rra
			rr	e
			ld	d,a			;DE = resulting MAPY * 64

			ld	b,(ix+SP_MAPX)		;Now do the same with MAPX and INTRAX
			ld	a,(ix+SP_INTRAX)
			add	a,l
			jp	m,ga_negx
ga_incx:		cp	8
			jr	c,ga_noincx
			inc	b
			sub	8
			jr	ga_incx
ga_negx:		dec	b
			add	a,8
			jp	m,ga_negx
ga_noincx:		ld	(ga_intrax),a		;Store INTRAX
			ld	a,b
			and	63
			add	a,e
			ld	e,a			;DE=MAPY*64+MAPX
			ld	hl,mapa
			add	hl,de			;HL=map position
			ld	a,(hl)			;A = tilenumber
			and	63			;remove higher bits
			jr	z,ga_voidme		;Blank floor, don't check
			cp	52
			jr	nc,ga_voidme		;CEZ and lines floor, don't check
			add	a,a			;Duplicate
			ld	e,a
			ld	d,tilelist/256		;DE points to tile
			ld	a,(de)
			ld	l,a
			inc	e
			ld	a,(de)
			ld	h,a			;HL= tile position, at (0,0)

			ld	a,(ga_intray)		;Get vertical value
			rla
			rla
			rla				;8*vertical
			ld	e,a
			ld	a,(ga_intrax)
			add	a,e			;plus horizontal
			ld	e,a

			ld	d,0		
			add	hl,de			;HL=attribute position
			ld	a,(hl)			;A=attribute
			ret
ga_voidme:		ld	a,42			;Floor, no penalization
			ret

;***************
;checkcollision: Test which car goes before the other, and check if they collide
;***************
;IX points to one car, IY points to another
checkcollision:		ld	a,(ix+SP_TIMESTAMPLAP)
			cp	(iy+SP_TIMESTAMPLAP)
			jr	c,iyhigher
			jr	nz,ixhigher		;Compare number of laps. The highest is placed before.
			ld	a,(ix+SP_TIMESTAMPCP)
			cp	(iy+SP_TIMESTAMPCP)
			jr	c,iyhigher
			jr	nz,ixhigher		;Compare checkpoint. The highest is placed before.
			ld	a,(ix+SP_TIMESTAMP)
			cp	(iy+SP_TIMESTAMP)
			jr	c,ixhigher
			jr	nz,iyhigher		;Compare timestamp (high). The lowest is placed before.
			ld	a,(ix+SP_TIMESTAMP+1)
			cp	(iy+SP_TIMESTAMP+1)
			jr	c,ixhigher
			jr	nz,iyhigher		;Compare timestamp (low). The lowest is placed before.
			ld	a,(ix+SP_COLORMARKER)
			cp	(iy+SP_COLORMARKER)
			jr	c,ixhigher		;Compare color. Fixed order magenta->blue->yellow->cyan
iyhigher:		inc	(iy+SP_RELPOS)		;Increase relative position of the car pointed by IY
			jr	donepositioning
ixhigher:		inc	(ix+SP_RELPOS)		;Increase relative position of the car pointed by IX
donepositioning:	ld	a,(ix+SP_STATE)
			and	(iy+SP_STATE)
			and	2
			ret	nz			;Don't check if they're both already colliding

			bit	4,(ix+SP_STATE)
			ret	nz			;Exit if first car is inactive

			bit	4,(iy+SP_STATE)
			ret	nz			;Exit if second car is inactive

			ld	a,(ix+SP_MAPY)
			sub	(iy+SP_MAPY)		;Compare vertical map positions
			jr	c,cc_negatifo_y
			cp	2			;Check positive value (IY car above IX car)
			jr	c,cc_noextrem_y
			cp	63
			ret	nz			;Exit if vertical positions are far
			sub	64
			jr	cc_noextrem_y
cc_negatifo_y:		cp	-2			;Check negative value (IY car below IX car)
			jr	nc,cc_noextrem_y
			cp	-63
			ret	nz			;Exit if vertical positions are far
			add	a,64
cc_noextrem_y:		add	a,a			;Vertical position is near
			add	a,a
			add	a,a
			add	a,(ix+SP_INTRAY)
			sub	(iy+SP_INTRAY)		;Calculate vertical distance in characters
			add	a,2			;Add 2, now the range we're looking for (-2 to 2) becomes (0 to 4)
			cp	5			
			ret	nc			;No collision if we're not in the range
			ld	e,a
			add	a,a
			add	a,a
			add	a,e
			ld	e,a			;E=(diffY+2)*5

			ld	a,(ix+SP_MAPX)
			sub	(iy+SP_MAPX)		;Compare horizontal map positions
			jr	c,cc_negatifo_x
			cp	2			;Check positive value (IY car to the left of IX car)
			jr	c,cc_noextrem_x
			cp	63
			ret	nz			;Exit if horizontal positions are far
			sub	64
			jr	cc_noextrem_x
cc_negatifo_x:		cp	-2			;Check negative value (IY car to the right of IX car)
			jr	nc,cc_noextrem_x
			cp	-63
			ret	nz			;Exit if horizontal positions are far
			add	a,64
cc_noextrem_x:		add	a,a			;Horizontal position is near
			add	a,a
			add	a,a
			add	a,(ix+SP_INTRAX)	;Calculate horizontal distance in characters
			sub	(iy+SP_INTRAX)		;Add 2, now the range we're looking for (-2 to 2) becomes (0 to 4)
			add	a,2
			cp	5		
			ret	nc			;No collision if we're not in the range
			add	a,e			;A = (diffX+2)+((diffY+2)*5)
			or	128			;Add 128
			ld	e,a
			ld	d,tablagiros/256	;DE = colis2[diffy][diffx]
			ld	a,(de)			;Take value from table "colis2" (25 values)
			rr	a			;Get first bit starting from the right
			jr	z,cc_crash		;If there was a 0 in the table, there's a collision for sure, no need to keep checking
			ld	(cc_checkit+1),a	;Self-modifying code BIT n,(hl)
			jr	nc,cc_noswapped		;If the bit is 0, no need to swap positions when comparing the cars
			ld	a,(iy+SP_FRAME)
			rla
			rla
			rla
			and	240			;Put the direction (bits 4-1 of frame) of car pointed by IY into bits 7-4
			ld	l,a			;Keep in register L
			ld	a,(ix+SP_FRAME)	
			jr	cc_continue
cc_noswapped:		ld	a,(ix+SP_FRAME)
			rla
			rla
			rla
			and	240			;Put the direction (bits 4-1 of frame) of car pointed by IX into bits 7-4
			ld	l,a			;Keep in register L
			ld	a,(iy+SP_FRAME)
cc_continue:		rra	
			and	15			;Put the direction (bits 4-1 of frame) of the other car into bits 3-0
			add	a,l			;Join both
			ld	l,a
			ld	h,colisiones/256	;Form address in table
			call	cc_checkit		;Check bit in the table
			ret	z			;Return if there's no collision

							;There's been a new collision!
cc_crash:		ld	a,25
			add	a,e
			ld	e,a
			ld	a,(de)			;Read from table colis3
			ld	d,0
			rra				;Get the rightmost bit
			jr	c,cc_swappedro		;If it's 1, let's swap the rotation
			ld	(ix+SP_COL_DIR),d
			inc	d
			ld	(iy+SP_COL_DIR),d
			dec	d
			jr	cc_noswappedro
cc_swappedro:		ld	(iy+SP_COL_DIR),d
			inc	d
			ld	(ix+SP_COL_DIR),d
			dec	d
cc_noswappedro:		rra				;Get another bit
			jr	nc,cc_norandomcra	;If no carry, the crash won't be random
			ld	a,(recording)
			cp	2
			jr	nz,cc_getrandom
			call	rec_get4bits		;If we're replaying, get the random number from the record stream
			ld	a,c
			jr	cc_norandomcra
cc_getrandom:		ld	a,r			;Read random number
			and	15			;Get 4 bits
			ld	c,a
			call	setrec4bits		;Store in the recording stream
			ld	a,c
cc_norandomcra:		ld	e,a

			ld	hl,fakespeeds
			add	hl,de
			ld	e,(hl)			;e=CRASH_DIRX
			ld	a,l
			sub	4
			ld	l,a
			ld	l,(hl)			;l=CRASH_DIRY

			ld	d,(ix+SP_DIRX)		;d=CAR1_DIRX
			ld	h,(iy+SP_DIRX)		;h=CAR2_DIRX
			ld	a,h
			add	a,e
			sra	a
			add	a,d
			sra	a
			add	a,e
			sra	a
			ld	(ix+SP_DIRX),a		;NEW CAR1_DIRX
			ld	a,d
			sub	e
			sra	a
			add	a,h
			sra	a
			sub	e
			sra	a
			ld	(iy+SP_DIRX),a		;NEW CAR2_DIRX
			ld	d,(ix+SP_DIRY)		;d=CAR1_DIRY
			ld	h,(iy+SP_DIRY)		;d=CAR2_DIRY
			ld	a,h
			add	a,l
			sra	a
			add	a,d
			sra	a
			add	a,l
			sra	a
			ld	(ix+SP_DIRY),a		;NEW CAR1_DIRY
			ld	a,d
			sub	l
			sra	a
			add	a,h
			sra	a
			sub	l
			sra	a
			ld	(iy+SP_DIRY),a		;NEW CAR2_DIRY
							;Directions are done

			ld	a,(ix+SP_COL_DUR)
			sra	a
			add	a,(ix+SP_SPEED)
			add	a,3
			ld	(ix+SP_SPEED),a		;Set speed for first car
			add	a,(iy+SP_SPEED)
			ld	e,a
			add	a,(ix+SP_SPEED)
			rra
			ld	(ix+SP_COL_DUR),a	;Set crash duration for first car

			ld	a,(iy+SP_COL_DUR)
			sra	a
			add	a,(iy+SP_SPEED)
			add	a,3
			ld	(iy+SP_SPEED),a		;Set speed for second car
			add	a,e
			rra
			
			ld	(iy+SP_COL_DUR),a	;Set crash duration for second car
			ld	a,(ix+SP_STATE)
			or	2
			ld	(ix+SP_STATE),a		;Set crash bit in state of first car
			ld	a,(iy+SP_STATE)
			or	2
			ld	(iy+SP_STATE),a		;Set crash bit in state of second car
			ret

;********
;control: Main control routine, checks the car's current state, and calls to its control routine if necessary
;********
control:		call	get_attr_self		;Get what we're standing on
			cp	42			;Floor?
			jr	nc,cont_nosemih		;Over it: We're not over stripes
							;  Yes: We're over the stripes, so car is half over the grass
			ld	a,(recording)
			cp	2
			jr	nz,control_randhie
			call	rec_getbit		;If we're replaying, read a bit from the recording stream
			jr	nc,cont_rechier		;Bit 0: slowdown
			jp	cont_nohierba		;Bit 1: no slowdown
control_randhie:	ld	a,r			;Get random number
			and	127			;Don't take high bit into consideration
			cp	85
			jp	c,cont_nohierba2	;Under 85 > no slowdown
			call	setrec0			;Over 84: slowdown > Set a 0 bit in the recording stream
cont_rechier:		ld	a,(ix+SP_SPEED)		;Get speed
			dec	a			;Decrease it
			jp	m,cont_nohierba		;No real effect if speeds turns negative after that decrement
			jr	cont_parado		;Slow down if it doesn't
			
cont_nohierba2: 	call    setrec1			;Set a 1 bit in the recording stream
                	jp      cont_nohierba		;And no effect
                	
cont_nohierba3: 	call    setrec0			;Set a 0 bit in the recording stream
                	ld      a,l			;Original speed
                	jp      cont_nohierba		;No slowdown
                	
cont_nosemih:		ld	a,(ix+SP_STATE)		;Get state	;
			jr	nz,cont_enlahierba
			and	254			;Erase grass bit from state
			ld	(ix+SP_STATE),a		;Update state
			jr	cont_nohierba
cont_enlahierba:	or	1			;We're on grass, set grass bit in state
			ld	(ix+SP_STATE),a		;Update state
			and	2
			jr	nz,cont_nohierba	;Don't slow down if we're crashing
			ld	a,(ix+SP_SPEED)		;Get speed
			and	a			;Check it
			jr	z,cont_nohierba		;Don't slow down if we're stopped
			ld	l,a			;Keep speed in register L
			dec	a			;Decrease it
			cp	2
			jr	z,cont_parado		;If it's 2, apply the slow down
			jr	nc,cont_sub1		
							;Speed is 1
			ld	a,(recording)
			cp	2
			jr	nz,cont_rand_enla2
			call	rec_getbit		;If we're replaying, read a bit from the recording stream
			ld	a,l			;Get original speed
			jr	nc,cont_nohierba	;No slowdown if bit was 0
                	jp      cont_sub1		;Slowdown if bit was 1
cont_rand_enla2:	ld      a,r			;Get random number
                	and     127
                	cp      45			;Over 44?
                	ld      a,l			;Get original speed
                	jr      nc,cont_nohierba3	;Over 44 > no slowdown
                					;Under 45
                	call    setrec1			;Set a 1 bit in the recording stream
                	ld      a,l			;Original speed
cont_sub1:		dec	a			;Decrease speed

cont_parado:		ld	(ix+SP_SPEED),a		;Update speed

cont_nohierba:		ld	a,(ix+SP_STATE)		;Get state
			ld	l,a			;Keep it in register L
			and	2			;Check bit 2 (crash)
			ld	a,l			;Restore state in A
			jr	nz,cont_crashcontrol	;If we're crashing, you're not in control
			and	4			;Check bit 4 (finished)
			jr	nz,cont_wincontrol	;If you've already finished the race, you're not in control either
			ld	l,(ix+SP_CONTROL)
			ld	h,(ix+SP_CONTROL+1)	;Get control routine in HL
			ld	de,control_apply	;Return point
			push	de			;Set return point in stack
			jp	(hl)			;Jump to this player's control routine

cont_crashcontrol:	and	253			;Clear collision bit
			ld	e,a			;Store just in case
			ld	a,(ix+SP_COL_DUR)	;Get collision duration
			dec	a			;Decrease it
			ld	(ix+SP_COL_DUR),a	;Update collision duration
			jr	nz,cont_notyet		;No change in state if we're still in collision
			
			ld	(ix+SP_STATE),e		;No more crashed
			ld	(ix+SP_SPEED),a		;Stop car...
			call	direction_update	;Update direction
			jr	cont_wincontrol		;Keep slowing down

cont_notyet:		ld	a,(ix+SP_COL_DIR)	;Check collision direction
			rra				;First bit
			jr	c,cont_rotright	
			call	inc2frame		;Rotate left if direction bit was 0
			jr	cont_wincontrol
cont_rotright:		call	dec2frame		;Rotate right if direction bit was 1
cont_wincontrol:	ld	a,(ix+SP_SPEED)		;Get speed
			and	a
			ret	z			;Return if already 0
			dec	a			;Decresase it
			ld	(ix+SP_SPEED),a		;Update speed
			ret

;*************
;control_play: Control of all cars when replaying the race.
;*************
control_play:   	call    rec_getbit
			pop	bc			;No control_apply
                	ld      a,(ix+SP_LASTCONTROL)
                	ld      c,a
                	jp      nc,control_apply2	;If bit read is 0, use the same control as last time
                	call    rec_get4bits		;Otherwise, take 4 new bits from the recording stream and use them
                	jp      control_apply3
                	
;***************
;control_clever: The most intelligent control routine. It looks ahead for grass, and uses all directions (22.5 precision instead of 45)
;***************
control_clever:		ld	l,(ix+SP_NEXTCHECK)
			ld	h,mapa_checks/256
			ld	a,(hl)		;Get left part of next checkpoint
			inc	l
			add	a,(hl)		;Add right
			rra			;Middle part
			sub	(ix+SP_MAPX)	;Check distance with car's current position
			ld	d,0		;D = 0 -> center, zero distance
			jr	z,clev_nohoriz	
			ld	d,1		;D = 1 -> left, positive distance
			jr	nc,clev_horiz
			ld	d,2		;D = 2 -> right, negative distance
			neg			;Absolute distance value
clev_horiz:		cp	32
			jr	c,clev_nohoriz
						;If distant is too high (>32), then it's the opposite direction
			ld	b,a		;Keep distance
			ld	a,d
			xor	3
			ld	d,a		;Reverse direction
			ld	a,64
			sub	b		;Adjust real distance = 64-dist
clev_nohoriz:		ld	b,a		;B = absolute distance X
						;Now let's repeat the same with vertical
			inc	l
			ld	a,(hl)		;Get up part of next checkpoint
			inc	l
			add	a,(hl)		;Add down
			rra			;Middle
			sub	(ix+SP_MAPY)	;Distance
			ld	e,0		;E = 0 -> center, zero distance
			jr	z,clev_novert	
			ld	e,6		;E = 6 -> up, positive distance
			jr	nc,clev_vert
			ld	e,3		;E = 3 -> down, negative distance
			neg			;positive

clev_vert:		cp	32
			jr	c,clev_novert
						;If distant is too high (>32), then it's the opposite direction
			ld	c,a
			ld	a,e
			xor	5		
			ld	e,a		;Reverse direction
			ld	a,64
			sub	c		;Adjust real distance = 64-dist
clev_novert:		ld	c,a		;C = absolute distance Y
			ld	hl,dirprefered-1
			ld	a,l
			add	a,d		;Add horizontal direction
			add	a,e		;Add vertical direction, clear carry
			ld	l,a
			ld	a,(hl)		;Get preprocessed prefered direction
			rra			;Get bit 0 of preprocessed prefered direction. 0 if diagonal, 1 if straight
			ld	l,a		;keep the rotated prefered direction in L
			jr	c,clev_recta	;Jump if straight direction

			rra			;Get bit 0 (used to be bit 1) of prefered direction.
			ld	l,a		;keep the rotated prefered direction in L
			jr	c,clev_diagalt

			ld	a,c		;bit was 0, that means signs of horizontal and vertical are different
			cp	b		;Compare Y distance to X distance
			jr	clev_cont1

clev_diagalt:		ld	a,b		;bit was 1, that means signs of horizontal and vertical are the same
			cp	c		;Compare X distance to Y distance
clev_cont1:		jr	z,clev_recta	;If they're the same, it's pure diagonal movement
						;  in other case, one of the directions has a bigger importance, so adjust
						;  and go for the 22.5 rotation frame that's needed
			jr	c,clev_decl
			inc	l		;Adjust diagonal direction to go for the dominant direction
			jr	clev_recta
clev_decl:		dec	l		;Adjust diagonal direction to go for the dominant direction
clev_recta:		ld	a,l
			ld	(prefdir),a	;Store definitive prefered direction
			ld	h,0
			ld	de,speeds
			add	hl,de		;HL=speeds (current prefered direction)

			call	checkgrass		;Look ahead in prefered direction
			jr	nz,clev_passnow		;If there's no grass, go for it
			call	checkgrass		;Look ahead turning 22.5 to the right from prefered direction
			jr	nz,clev_passright	;If there's no grass, go for it
			dec	l
			dec	l
			dec	l
			call	checkgrass		;Look ahead turning 22.5 to the left from prefered direction
			jr	nz,clev_passleft	;If there's no grass, go for it

			ld	c,14			;Grass is everywhere, slow down!
			jr	clev_slowdown

clev_passright:		ld	a,(prefdir)
			inc	a
			and	15
			ld	(prefdir),a		;Adjust prefered direction to the right
clev_passnow:		ld	c,15			;Don't do anything default control (all 4 bits at 1)
clev_slowdown:		ld	a,(prefdir)
			ld	b,a			;Keep prefered direction in A
			ld	a,(ix+SP_FRAME)		;Get current frame
			and	a			;Clear carry flag
			rra				;Divide by 2
			sub	b			;Calculate difference between prefered and current direction
			jr	z,clev_straight		;if zero, we're already on track -> keep straight
			and	15			;Keep only the lower bits
			cp	8			;check difference
			jr	c,clev_turnleft		;below 8, turn left
			ld	c,11			;control: turn right (clear bit 2)
			jr	clev_straight
clev_turnleft:		ld	c,7			;control: turn left (clear bit 4)
clev_straight:		ld	a,c
			ld	e,15			;Desired speed = 15 (maximum)
			cp	15			;No movement?
			jr	z,clev_nomaxspeed	;If there's no movement, continue
			ld	e,10			;Desired speed = 10 when turning
			ld	a,c
			or	1
			ld	c,a			;Make sure bit 0 is set?
clev_nomaxspeed:	jp	cb_nomaxspeed		;Keep moving in control_blind routine

clev_passleft:		ld	a,(prefdir)
			dec	a
			and	15
			ld	(prefdir),a		;Adjust prefered direction to the left
			jp	clev_passnow

;***************
;control_varied: control routine for all CPUs, it chooses another control routine depending on a random value and the constants set by the difficult level
;***************
control_varied:		ld	a,r
			and	127			;Take 7 bits
			cp	(ix+SP_IARAND1)	
			jp	nc,control_blind	;If the value read is below IARAND1, use the blind routine 1
			cp	(ix+SP_IARAND2)
			jp	nc,control_rand2	;If the value read is below IARAND2, use the totally random routine
			cp	(ix+SP_IARAND3)
			jp	nc,control_clever	;If the value read is below IARAND3, use the clever routine
							;Otherwise, use the following blind routine 2

;***************
;control_blind2: another stupid control routine, is fine with any portion of the checkpoint, instead of going for the centre
;***************
control_blind2: 	ld	a,(ix+SP_STATE)		;Get state
			dec	a			;1? (over grass)
			jp	z,control_blind		;If we're over grass, use the control_blind function instead
			ld	l,(ix+SP_NEXTCHECK)	;Get next checkpoint address
			ld	h,mapa_checks/256
			ld	a,(ix+SP_MAPX)		;Get current X position
			cp	(hl)			;Compare to left part of checkpoint
			inc	hl			;point to right part, without touching the flags
			ld	e,64			;64 for right
			jr	c,cb2_horizpassedka	;If carry, it seems like we need to go right, go to check distance
			ld	e,0			;0 for center
			jr	z,cb2_horizpassed	;If zero, horizontal is done 
			cp	(hl)			;Compare to the right part of checkpoint
			jr	z,cb2_horizpassed
			jr	c,cb2_horizpassed	;If zero or below, horizontal is done 
							;Seems to be left
			ld	e,32			;32 for left
			sub	(hl)
			cp	32			;Check if distance is over 32
			jr	nc,cb2_horizreverse	;Reverse if distance is over 32
			jr	cb2_horizpassed	

cb2_horizpassedka:	sub	(hl)
			add	a,32			;Check if distance is over 32
			jr	c,cb2_horizpassed	;No -> horizontal value is ok
cb2_horizreverse:	ld	a,e
			xor	96			;Reverse left<->right
			ld	e,a
cb2_horizpassed:	inc	l			;Point to up part
			ld	a,(ix+SP_MAPY)		;Get current Y position
			cp	(hl)			;Compare to the up part of checkpoint
			inc	hl			;Point to down part, without touching the flags
			ld	d,192			;192 for down
			jr	c,cb2_vertpassedka	;If carry, it seems like we need to go down, go to check distance
			ld	d,0			;0 for center
			jr	z,cb2_vertpassed	;If zero, vertical is done 
			cp	(hl)			;Compare to the down part of checkpoint
			jr	z,cb2_vertpassed
			jr	c,cb2_vertpassed	;If zero or below, horizontal is done 
							;Seems to be up
			ld	d,96			;96 for up
			sub	(hl)
			cp	32			;Check if distance is over 32
			jr	nc,cb2_vertreverse	;Reverse if distance is over 32
			jr	cb2_vertpassed
cb2_vertpassedka:	sub	(hl)
			add	a,32			;Check if distance is over 32
			jr	c,cb2_vertpassed
cb2_vertreverse:	ld	a,d
			xor	160			;Reverse up<->down
			ld	d,a
cb2_vertpassed:		ld	a,d			;Get vertical component
			add	a,e			;Add horizontal constant
			jp 	cb_novert2		;Go to look up the control in the table in control_blind function

;**************
;control_blind: stupid control routine, goes blindly to the center of the next checkpoint
;**************
control_blind:  	ld	l,(ix+SP_NEXTCHECK)	;Get next checkpoint address
			ld	h,mapa_checks/256
			ld	a,(hl)			;Get left part of checkpoint
			inc	l
			add	a,(hl)			;Add right part
			rra				;Medium
			sub	(ix+SP_MAPX)		;Substract current position
			jr	z,cb_nohoriz		;if zero, we're there, 0 for center
			jr	c,cb_noright		;If carry, we must probably go left
							; no carry, we must probably go right
			cp	32			;difference over 32?
			jr	nc,cb_yesleft		;Yes: so it's left instead
							; No: it's indeed right
cb_yesright:		ld	a,64			;64 for right
			jr	cb_nohoriz		;Horizontal calculation done

cb_noright:		add	a,32			;difference over 32?
			jr	nc,cb_yesright		;Yes: so it's right instead
							; No: it's indeed left
cb_yesleft:		ld	a,32			;32 for left

cb_nohoriz:		ld	e,a			;Keep horizontal constant in E register
			inc	l			
			ld	a,(hl)			;Get up part of checkpoint
			inc	l
			add	a,(hl)			;Add down part
			rra				;Medium
			sub	(ix+SP_MAPY)		;Substract current position
			jr	z,cb_novert		;if zero, we're there, 0 for center
			jr	c,cb_nodown		;If carry, we must probably go up
							; no carry, we must probably go down
			cp	32			;difference over 32?
			jr	nc,cb_yesup		;Yes: so it's up instead
							; No: it's indeed down
cb_yesdown:		ld	a,192			;192 for down
			jr	cb_novert
cb_nodown:		add	a,32			;difference over 32?
			jr	nc,cb_yesdown		;Yes: so it's down instead
							; No: it's indeed up
cb_yesup:		ld	a,96			;96 for up
cb_novert:		add	a,e			;Add horizontal constant
cb_novert2:		add	a,(ix+SP_FRAME)		;Add current frame
			rra				;Divide by 2
			ld	l,a			;and we get the lookup table address
			ld	h,tablagiros/256	;complete address
			ld	a,(hl)			;Get control from table
			ld	e,10			;Desired speed: 10
			cp	15			;No movement?
			ld	c,a
			jr	nz,cb_nomaxspeed	;If there was no movement
			ld	e,15
cb_nomaxspeed:		ld	a,(ix+SP_SPEED)		;Get speed
			cp	e			;Compare to desired speed
			ret	z			;Return if we're already there
			jr	nc,cb_speeddown		;If we're over desired speed, slow down (set bit 0 of control to 0, decreasing once)
							;Otherwise, accelerate (set bit 1 of control to 0, decreasing twice)
			dec	c
cb_speeddown:		dec	c
			ret

;**************
;control_apply: Move the cars depending on the value given by the control routine in register C
;**************
control_apply:  	ld      a,c
                	cp      (ix+SP_LASTCONTROL)	;Compare to last control value
                	jr      nz,contap_recordit
                	call    setrec0			;If it's equal, insert a 0 in the recording stream
                	jr      control_apply2		;Then apply it
contap_recordit:	call    setrec1			;If it's different, insert a 1 in the recording stream
                	call    setrec4bits		;And then store the control bits themselves
control_apply3: 	ld      (ix+SP_LASTCONTROL),c	;Store new last control
control_apply2:		ld	a,(ix+SP_SPEED)		;Get speed
			and	a
			ld	e,a
			ld	d,0
			ld	hl,girospeeds
			add	hl,de			;HL points to rotation resistance (depends on speed)
			bit	3,c			;Check Right
			jr	nz,norotateright
			ld	a,(ix+SP_SUBGIRO)
			sub	(hl)			;Check resistance to rotation
			ld	(ix+SP_SUBGIRO),a	;Update turning state
			jr	c,norotateleft		;If there's carry, there's no rotation
			call	dec2frame		;Decrease frame by 2 units
			call	direction_update	;Update direction
			jr	norotateleft
norotateright:		bit	2,c			;Check Left
			jr	nz,norotateleft
			ld	a,(ix+SP_SUBGIRO)
			sub	(hl)			;Check resistance to rotation
			ld	(ix+SP_SUBGIRO),a	;Update turning state
			jr	c,norotateleft		;If there's carry, there's no rotation
			call	inc2frame		;Increase frame by 2 units
			call	direction_update	;Update direction
norotateleft:		bit	0,c			;Check Down
			jr	nz,nospeeddown
			ld	a,(ix+SP_SPEED)		;Get speed
			and	a			;Already zero?
			jr	z,nospeeddown
			dec	a			;No > decrease it
			ld	(ix+SP_SPEED),a		;     and keep it
nospeeddown:		bit	1,c			;Check Up
                	ret	nz
                	ld	a,(ix+SP_SUBSPEED)	;Get subspeed value
                	add	a,(ix+SP_ACCELRATE)	;Try to accelerate
                	ld	(ix+SP_SUBSPEED),a	;Store subspeed value
                	ret	nc			;No carry, no accelaration
                	ld	a,(ix+SP_SPEED)		;Get speed
                	cp	15			;Already max speed?
                	ret	nc			;Yes > exit
                	inc	a			;No > increase it
                	ld	(ix+SP_SPEED),a		;     and keep it
control_dummy:		ret

;**************
;control_rand2: Control routine. Use 4 lower bits of random value read as control value.
;**************
;control_rand:		ld	a,r			;Get random value
control_rand2:		and	15			;Get 4 low bits
			ld	c,a			;Keep it in C
			ret				;Return
			;jp	control_apply

;***************
;movecar_scroll: Update the car position, and scroll the background if necessary
;***************
;IX=car pointer
;IY=camera pointer
movecar_scroll:		ld	b,(ix+SP_SPEED)		;Use speed as a counter
			ld	a,b
			and	a
			ret	z			;Return if speed is 0
			push	bc			;Keep counter for vertical movement
			ld	a,(ix+SP_DIRY)		;Get vertical speed direction
			and	a			;Check sign
			ld	h,a
			ld	a,(ix+SP_SUBY)		;Get vertical subspeed variable
			jp	m,msc_up
msc_down:		add	a,h			;Add speed to subspeed variable
			jr	nc,msc_down_no		;No carry means no movement
			ld	c,a			;Keep subspeed
			push	hl
			push	bc
			call	incy			;Move down
			call	scrollbg_down		;Scroll down
			pop	bc
			pop	hl
			ld	a,c			;Restore subspeed
msc_down_no:		djnz	msc_down		;Do it B times
			jp	msc_horiz		;Then go to horizontal movement
msc_up:			add	a,h			;Add speed to subspeed variable
			jr	c,msc_up_no		;No carry means no movement
			ld	c,a			;Keep subspeed
			push	hl
			push	bc
			call	decy			;Move up
			call	scrollbg_up		;Scroll up
			pop	bc
			pop	hl
			ld	a,c			;Restore subspeed
msc_up_no:		djnz	msc_up			;Do it B times
msc_horiz:		ld	(ix+SP_SUBY),a		;Store final vertical subspeed
			pop	bc			;Restore counter
			
			ld	a,(ix+SP_DIRX)		;Do the same for horizontal speed
			and	a
			ld	h,a
			ld	a,(ix+SP_SUBX)
			jp	m,msc_left
msc_right:		add	a,h
			jr	nc,msc_right_no
			ld	c,a
			push	hl
			push	bc
			call	incx
			call	scrollbg_right
			pop	bc
			pop	hl
			ld	a,c
msc_right_no:		djnz	msc_right
			jp	msc_end
msc_left:		add	a,h
			jr	c,msc_left_no
			ld	c,a
			push	hl
			push	bc
			call	decx
			call	scrollbg_left
			pop	bc
			pop	hl
			ld	a,c
msc_left_no:		djnz	msc_left
msc_end:		ld	(ix+SP_SUBX),a
			ret

;********
;movecar: Just update the car position
;********
;IX=car pointer
movecar:		ld	b,(ix+SP_SPEED)		;Use speed as a counter
			ld	a,b
			and	a
			ret	z			;Return if speed is 0
			push	bc			;Keep counter for vertical movement
			ld	a,(ix+SP_DIRY)		;Get vertical speed direction
			and	a			;Check sign
			ld	h,a
			ld	a,(ix+SP_SUBY)		;Get vertical subspeed variable
			jp	m,move_up
move_down:		add	a,h			;Add speed to subspeed variable
			jr	nc,move_down_no		;No carry means no movement
			ld	c,a			;Keep subspeed
			call	incy			;Move down
			ld	a,c			;Restore subspeed
move_down_no:		djnz	move_down		;Do it B times
			jp	move_horiz		;Then go to horizontal movement

move_up:		add	a,h			;Add speed to subspeed variable
			jr	c,move_up_no		;No carry means no movement
			ld	c,a			;Keep subspeed
			call	decy			;Move up
			ld	a,c			;Restore subspeed
move_up_no:		djnz	move_up			;Do it B times

move_horiz:		ld	(ix+SP_SUBY),a		;Store final vertical subspeed
			pop	bc			;Restore counter

			ld	a,(ix+SP_DIRX)		;Do the same for horizontal speed
			and	a
			ld	h,a
			ld	a,(ix+SP_SUBX)
			jp	m,move_left
move_right:		add	a,h
			jr	nc,move_right_no
			ld	c,a
			call	incx
			ld	a,c
move_right_no:		djnz	move_right
			jp	move_end
move_left:		add	a,h
			jr	c,move_left_no
			ld	c,a
			call	decx
			ld	a,c
move_left_no:		djnz	move_left
move_end:		ld	(ix+SP_SUBX),a
			ret

;************
;brujula_rut: returns the address of the compass character for the car pointed in IX
;************
brujula_rut:  		ld	l,(ix+SP_NEXTCHECK)
			ld	h,mapa_checks/256
			ld	a,(hl)
			inc	l
			add	a,(hl)
			rra			;get the middle point of both horizontal positions in checkpoints table

			sub	(ix+SP_MAPX)	
			jr	z,bru_nohoriz	;0 for center
			jr	c,bru_noright
			cp	32
			jr	nc,bru_yesleft
bru_yesright:		ld	a,2		;2 for right
			jr	bru_nohoriz
bru_noright:		add	a,32
			jr	nc,bru_yesright
bru_yesleft:		ld	a,1		;1 for left
bru_nohoriz:		ld	e,a		;Keep value in E
			inc	l
			ld	a,(hl)
			inc	l
			add	a,(hl)
			rra			;get the middle point of both vertical positions in checkpoints table
			sub	(ix+SP_MAPY)
			jr	z,bru_novert	;0 for center
			jr	c,bru_nodown
			cp	32
			jr	nc,bru_yesup
bru_yesdown:		ld	a,6		;6 for down
			jr	bru_novert
bru_nodown:		add	a,32
			jr	nc,bru_yesdown
bru_yesup:		ld	a,3		;3 for up
bru_novert:		add	a,e		;Add horizontal value
			add	a,104		;Add constant
			ret

;**********		
;inc2frame: advance 1 frame
;**********
inc2frame:		ld	de,9		;Graphic advance
			ld	a,(ix+SP_FRAME)	;Get current frame
			add	a,2		;Increase it
			cp	32		;Check if we've passed the end
			jr	c,inc2frame_doit
			sub	32		;In that case, go back
			ld	de,-135		;New graphic advance -15*9
inc2frame_doit:		ld	(ix+SP_FRAME),a	;Update frame
			ld	h,(ix+SP_GRAPH+1)
			ld	l,(ix+SP_GRAPH)
			add	hl,de
			ld	(ix+SP_GRAPH+1),h
			ld	(ix+SP_GRAPH),l	;Update graphic pointer for the car
			ret

;**********
;dec2frame: decrease 1 frame
;**********
dec2frame:		ld	de,-9		;Graphic advance
			ld	a,(ix+SP_FRAME)	;Get current frame
			dec	a
			dec	a		;Decrease it
			jp	p,inc2frame_doit;Check if we've passed the start
			add	a,32		;In that case, advance to the last
			ld	de,135		;New graphic advance
			jr	inc2frame_doit


;***********
;checkgrass: Check what lies ahead in a certain direction
;***********
checkgrass:		ld	a,(hl)		;Get vertical speed
			sra	a
			sra	a		;Divide by 4
			ld	e,a		;Keep vertical Displacement in register E
			ld	a,l
			sub	4
			ld	l,a		;HL=HL-4
			ld	a,(hl)		;Get horizontal speed
			sra	a
			sra	a		;Divide by 4
			ld	d,a		;Keep vertical Displacement in register D
			ld	a,l
			add	a,5		;HL=HL+5
			ld	l,a		;HL=speeds (current prefered direction+1)
			push	hl		;Keep HL
			ld	l,d		;Vertical Displacement in register L
			call	get_attr	;Get attribute
			pop	hl		;Restore HL
			and	56		;BUG: This is only valid for Spectrum
			cp	32		;BUG: The clever routine won't detect grass correctly -> needs to do cp 43, and check with C instead of NZ
			ret

;*****
;incx: increase x position
;*****
incx:		ld	a,(ix+SP_INTRAX)	;19  | 		Get INTRAX value
		inc	a			; 4  |		Increase it
		and	7			; 7  |		Keep it in 0-7 range
		ld	(ix+SP_INTRAX),a	;19  |		Update it
		ret	nz			;11,5|60 	If it's not 0, return
						;-6 |		If it's 0, we must increase MAPX
		ld	a,(ix+SP_MAPX)		;19 |		Get MAPX value
		inc	a			; 4 |		Increase it
		and	63			; 7 |		Keep it in 0-63 range
		ld	(ix+SP_MAPX),a		;19 |		Update it
incx_nomap:	ret				;10 | 53

;*****		
;decx: decrease x position
;*****
decx:		ld	a,(ix+SP_INTRAX) ;Get INTRAX value
		dec	a		 ;Decrease it
		jp	p,decx_nomap
					 ;If it becomes negative
		ld	a,(ix+SP_MAPX)   ;Get MAPX
		dec	a		 ;Decrease it
		and	63		 ;Keep it in 0-63 range
		ld	(ix+SP_MAPX),a	 ;Update it
		ld	a,7		 ;Value 7 for INTRAX
decx_nomap:	ld	(ix+SP_INTRAX),a ;Update it
		ret

;*****
;incy: increase y position
;*****
incy:		ld	a,(ix+SP_INTRAY)	;19  | 		Get INTRAY value
		inc	a			; 4  |		Increase it
		and	7			; 7  |		Keep it in 0-7 range
		ld	(ix+SP_INTRAY),a	;19  |		Update it
		ret	nz			;11,5|60	If it's not 0, return
						;-6 |		If it's 0, we must increase MAPY
		ld	a,(ix+SP_MAPY)		;19 |		Get MAPY value
		inc	a			; 4 |		Increase it
		and	63			; 7 |		Keep it in 0-63 range
		ld	(ix+SP_MAPY),a		;19 |		Update it
incy_nomap:	ret				;10 | 53

;*****
;decy: decrease y position
;*****
decy:		ld	a,(ix+SP_INTRAY) ;Get INTRAY value
		dec	a		 ;Decrease it
		jp	p,decy_nomap	 
					 ;If it becomes negative
		ld	a,(ix+SP_MAPY)   ;Get MAPY
		dec	a		 ;Decrease it
		and	63		 ;Keep it in 0-63 range
		ld	(ix+SP_MAPY),a	 ;Update it
		ld	a,7		 ;Value 7 for INTRAY
decy_nomap:	ld	(ix+SP_INTRAY),a ;Update it
		ret


;*************
;control_key2: Keyboard 2 control routine
;*************
control_key2:           ld      hl,keys2	;Point to keyboard 2 keys
                        jr      control_keys
;*************
;control_key3: Keyboard 3 control routine
;*************
control_key3:           ld      hl,keys3	;Point to keyboard 3 keys
                        jr      control_keys
;*************
;control_key4: Extra keys control routine
;*************
control_key4:           ld      hl,keys4	;Point to keyboard 4 keys
                        jr      control_keys

;*************
;control_key1: Keyboard 1 control routine
;*************
control_key1:           ld      hl,keys1	;Point to keyboard 1 keys
control_keys:           ld      b,4		;Number of keys
control_loop:           ld      a,(hl)		;Get row (+ high bits of port)
                        out     (#AA),a		;Output to select row
                        inc     hl
                        in      a,(#A9)         ;read row into A
                        and     (hl)		;Isolate key bit
                        inc     hl
                        add     a,(hl)		;Add negative: if key isn't pressed, carry will be 1
                        inc     hl
                        rl      c		;Enter bit into C
                        djnz    control_loop	;Repeat for all keys
                        ret

;*************
;control_joy1: Joystick 1 control routine
;*************
control_joy1:		ld	a,15
			out	($A0),a			;Select register 15
			in	a,($A2)			;Read register 15
			and	175			;Select joystick 0
			or	3			;Select joystick 0
			jr	control_joycommon
			
;*************
;control_joy1: Joystick 1 control routine
;*************
control_joy2:		ld	a,15
			out	($A0),a			;Select register 15
			in	a,($A2)			;Read register 15
			and	223
			or	78			;Select joystick 1
control_joycommon:	out	($A1),a			;Write register 15
			ld	a,14
			out	($A0),a			;Select register 14
			in	a,($A2)			;Read register 14
					;Here: A = xxFFRLDU CF=?
			rrca		;          UxxFFRLD CF=U
			rrca		;          DUxxFFRL CF=D
			rrca		;          LDUxxFFR CF=L
			rrca		;          RLDUxxFF CF=R (get)
			ld	b,a
			rl	c
			rlca		;	   LDUxxFFR CF=R
			rlca		;	   DUxxFFRL CF=L (get)
			rl	c
			rlca		;	   UxxFFRLD CF=D
			rlca		;	   xxFFRLDU CF=U (get)
			rl	c
			rrca		;	   UxxFFRLD CF=U
			rrca		;	   DUxxFFRL CF=D (get)
			rl	c
			and	8	;Fire 2?
			jr	nz,cjoy_nofire2
			res	0,c	;Fire 2 = Brake
			ret
cjoy_nofire2:		ld	a,b
			rrca		;	   FRLDUxxF CF=F1 (get)
			ret	c
			res	1,c	;Fire 1 = Accelerate
			ret

;*************
;control_kemp: Do nothing, kempston doesn't exist in MSX!
;*************
control_kemp:		ld	c,15
			ret		;Do nothing

;********
;isr_wyz: WYZPlayer interrupt routine
;********
isr_wyz:		ld	a,(current_sound)
                	and	1			;Check if sound is enabled
                	jr	z,isr_wyz_dosound
                	call	SILENCIAPLAYER		;If not, silence PSG
                	jp	int_end			;Exit from interrupt
isr_wyz_dosound:	ld	a,(int_ticks)		;Get number of interrupt ticks
			cp	6
			jp	z,int_end		;If it's 6, don't play any sound
			CALL    PLAY			;Prepare next notes
			LD	HL,PSG_REG
			LD	DE,PSG_REG_SEC
			LD	BC,15
			LDIR				;Copy PSG registers into secondary set
                	CALL    REPRODUCE_SONIDO
                	CALL    ROUT			;Play the music
                	jp	int_end			;Exit from interrupt

;*********
;int_menu: Menu interrupt routine, does rainbow effect and calls PT3 player.
;*********
int_menu:		ld	a,(rainbow_on)		;Check if rainbow is enabled
			and	a
			jp	z,norainbow		;If not, skip it
			ld	a,(VDPportWRITE)	;Get VDP write port
			ld	c,a
			ld	a,8			;Color address for first character, low byte
			out	(c),a			;Set low byte
			ld	a,96			;Color address for first character, high byte
			nop
			nop				;Small pause
			out	(c),a			;Set high byte
			dec	c			;Data port
			ld	de,28*257		;28 characters up, 28 down
rainfirst:		ld	hl,rainbow_data		;Point to the start rainbow colors
			outi
			outi
			outi
			outi
			outi
			outi
			outi
			outi				;Output all colors
			dec	e
			jr	nz,rainfirst		;Do it for the 28 higher characters
rainsec:		ld	hl,rainbow_data+7	;Point to the end of rainbow colors
			outd
			outd
			outd
			outd
			outd
			outd
			outd
			outd				;Output all colors backwards for the lower characters
			dec	d			
			jr	nz,rainsec		;Do it for the 28 lower characters
			ld	hl,(rainbow_rompoint)	;Get ROM address to get the colors
			ld	de,rainbow_data
			ld	b,7
			ld	a,(hl)			;Get ROM value
			and	240			;Keep only the higher bits, and set background black
			ld	(de),a			;Store it in rainbow_data buffer
			inc	hl			;Increase ROM address
			inc	de			;Increase destination
			set	7,h
			res	6,h			;Keep ROM address in page 2
			ld	(rainbow_rompoint),hl	;Store new ROM address
			ld	b,7			
rain_update:		ld	a,(hl)			;Get ROM value
			and	240			;Keep only the higher bits, and set background black
			ld	(de),a			;Store it in next position of rainbow_data 
			inc	hl			;Increase ROM address
			inc	de			;Increase destination
			djnz	rain_update		;Repeat for the remaining 7 bytes.
norainbow:		ld	a,(msxHZ)		;Get HZs
			cp	6
			jr	nz,mainint_menu
			ld	a,(int_ticks)		;If we're on a 60Hz model
			sub	6
			jp	z,int_endticks		;Don't play music if we're in tick number 6
mainint_menu:		call	PT3_PLAY		; Calculates PSG values for next frame
			call	PT3_ROUT		; Write values on PSG registers
			jp	int_end
int_endticks:		ld	(int_ticks),a		;Update tick counter from 6 to 0
			jp	int_end			;Exit from interrupt

;******************
;dofirststepsframe: Do the first steps of next frame generation. It's executed right after dumping the screen in 50Hz, and right before dumping the screen in 60Hz
;******************
dofirststepsframe:	ld	a,(tableadvance)	;If any car arrived in the previous frame, advance the number of cars that arrived in the tablepointer
			ld	b,a
			ld	a,(tablepointer)
			add	a,b
			ld	(tablepointer),a
			xor	a
			ld	(tableadvance),a	;Default no advance in tablepoints

			ld	de,spritesbuffer+15
			ld	hl,spritesbuffer+11
			ld	bc,12
			lddr				;Move previous sprites down

			ld	b,4			;Number of cars
			ld	ix,car1			;First car
main_cars:		push	bc			;Keep number of cars
			ld	a,(recording)
			cp	2			;Check if we're recording
			jr	nz,main_checkffnormal
			bit	7,(ix+SP_STATE)		;We're recording, check if we're a human car
			jr	z,main_nofastforwardyet	;  No: don't check for fastforward
			jr	main_tryfastforward	;  Yes: check for fastforward

main_checkffnormal:	bit	6,(ix+SP_STATE)		;We're not recording, check if we've arrived
			jr	z,main_nofastforwardyet	;  No: don't check for fastforward
			ld	a,(humansarrived)	;  Yes, check if all humans have arrived
			ld	d,a
			ld	a,(humans)
			cp	d			;All humans in?
			jr	nz,main_nofastforwardyet ; No: don't check for fastforward yet
							;  Yes: check for fastforward
							
main_tryfastforward:	ld	l,(ix+SP_CONTROLBAK)	
			ld	h,(ix+SP_CONTROLBAK+1)
			ld	de,main_fastforwardka	;Return value
			push	de			;In the stack
			jp	(hl)			;Call the backup control routine (the main control routine might be replaced by the replay one)

main_fastforwardka:	bit	1,c			;Check up control
			ld	a,10
			jr	z,main_dofastforward	;If pressed, do fast forward
			bit	0,c			;Check down control
			jr	nz,main_nofastforwardyet
			xor	a			;If pressed, don't do fast forward
main_dofastforward:	ld	(fastforward),a		;Update fastforward variable
main_nofastforwardyet:	ld	a,(minimap_cycle)	;Check if it's this car's turn to appear in minimap
			ld	e,a
			and	3			;Current turn in A
			cp	(ix+SP_CAMERA)		;Compare to our ID
			jr	nz,main_nominimap	;Jump if it's not our turn
			
			ld	a,(ix+SP_MAPX)		;Get horizontal position
			add	a,60			;Adjust
			ld	(spritesbuffer+1),a	;Store horizontal position
			ld	a,(ix+SP_MAPY)		;Get vertical position
			add	a,127			;Adjust
			ld	(spritesbuffer),a	;Store vertical position
			ld	a,(ix+SP_COLOR)		;Get color
			rlca
			rlca
			rlca
			rlca				;Swap ink & paper
			ld	(spritesbuffer+3),a	;Set color
			ld	a,98
			ld	(spritesbuffer+2),a	;Set sprite form
main_nominimap:		
			bit	4,(ix+SP_STATE)		;Check if the car is participating
			jp	nz,main_nocheck		;Go to next car if it's not.

			ld	a,(ix+SP_SPEED)
			and	a
			call	nz,movetyres		;If speed isn't 0, move the tyres of the car
			ld	a,(ix+SP_CAMERA)
			ld	iy,camera_left
			cp	(iy+CAM_CAR)
			jp	z,main_scrollyleft	;If current car is in left camera, go to main_scrollyleft
			ld	iy,camera_right
			cp	(iy+CAM_CAR)
			jp	nz,main_noscroll	;If current car isn't in right camera either, go to main_noscroll


			ld	a,(mode)		;Check category
			and	2
			jr	nz,main_brujula_right	;If category is JUNIOR or KARTS, put compass
			ld	a,(ix+SP_LAPS)		;If not, get number of laps
			add	a,16			;Turn into a character
			jr	main_nobrujula_right
main_brujula_right: 	call	brujula_rut		;Get current compass character
main_nobrujula_right:	ld	(lap_right),a		;Set character
			jr	main_scrollyright	;Go to move with scroll

main_scrollyleft:	ld	a,(mode)		;Check category
			and	2
			jr	nz,main_brujula_left	;If category is JUNIOR or KARTS, put compass
			ld	a,(ix+SP_LAPS)		;If not, get number of laps
			add	a,16			;Turn into a character
			jr	main_nobrujula_left
main_brujula_left:	call	brujula_rut		;Get current compass character
main_nobrujula_left:	ld	(lap_left),a		;Set character
			
main_scrollyright:	call	movecar_scroll		;Do car movement with scroll
			call	control			;Control car
			call	camerapanning		;Make the camera follow the action if necessary
			call	timeprint_car		;Paint the car's timer
			call	velocimetro		;Paint the speedometer
			jp	main_continue
main_noscroll:		call	movecar			;Do car movement without scroll
			call	control			;Control car

main_continue:		ld	l,(ix+SP_NEXTCHECK)	;Check if we've reached the next checkpoint
			ld	h,(ix+SP_NOVICE)	;For humans, JUNIOR and GP2 categories will check against novice checkpoints, KARTS and F1 agains the normal checkpoints
			ld	a,(ix+SP_MAPX)
			cp	(hl)
			jp	c,main_nocheck		;If the car's at the left of next checkpoint, we haven't reached it yet
			dec	a
			inc	l
			cp	(hl)
			jp	nc,main_nocheck		;If the car's at the right of next checkpoint, we haven't reached it yet
			ld	a,(ix+SP_MAPY)
			inc	l
			cp	(hl)
			jp	c,main_nocheck		;If the car's above the next checkpoint, we haven't reached it yet
			dec	a
			inc	l
			cp	(hl)
			jp	nc,main_nocheck		;If the car's below the next checkpoint, we haven't reached it yet
							;Checkpoint reached
			inc	l
			ld	(ix+SP_NEXTCHECK),l 	;Point to the next one
			ld	bc,(main_ticks)
			ld	a,l
			and	28			;For each 8 checkpoints, set a new timestamp
			jr	nz,main_checklast
			ld	a,(ix+SP_LAPS)
			ld	(ix+SP_TIMESTAMPLAP),a	;Lap for this timestamp
			ld	(ix+SP_TIMESTAMPCP),l	;Checkpoint for this timestamp
			ld	(ix+SP_TIMESTAMP+0),b	;Timestamp, high byte
			ld	(ix+SP_TIMESTAMP+1),c	;Timestamp, low byte
			ld	a,(msxHZ)
			cp	5
			ld	a,1			;Tell the marker to be updated
			jr	z,main_changeno60hz
			ld	a,4			;Tell the marker to be updated after another frame if we're in 60Hz
main_changeno60hz:	ld	(changemarker),a
main_checklast:		ld	a,l
			ld	l,0
			cp	(hl)			;Check if we passed the last one
			jp	nz,main_nocheck		;No? -> continue

			ld	l,1			;Last checkpoint, back to first
			ld	(ix+SP_NEXTCHECK),l	
			ld	a,(msxHZ)
			cp	5
			ld	a,1			;Tell the marker to be updated
			jr	z,main_changeno60hz2
			ld	a,4			;Tell the marker to be updated after another frame if we're in 60Hz
main_changeno60hz2:	ld	(changemarker),a
			ld	a,(ix+SP_LAPS)
			inc	a
			ld	(ix+SP_LAPS),a		;Increase number of laps
			ld	(ix+SP_TIMESTAMPLAP),a	;Lap for this timestamp
			ld	(ix+SP_TIMESTAMPCP),l	;Checkpoint for this timestamp
			ld	(ix+SP_TIMESTAMP+0),b	;Timestamp, high byte
			ld	(ix+SP_TIMESTAMP+1),c	;Timestamp, low byte
			ld	l,a
			dec	a
			jr	z,main_firstlap		;First lap?
							;No first lap
			ld	a,c
			sub	(ix+SP_LAPSTART+1)
			ld	c,a
			ld	a,b
			sbc	a,(ix+SP_LAPSTART+0)
			ld	b,a			;bc = ticks for this lap
		
			sub	(ix+SP_BESTLAP+0)	;best lap so far?
			jr	c,main_setnewbest	;Yes: go set it
			jr	nz,main_nofirstlap	;No: don't set it
							;Maybe: check the lower byte
			ld	a,c
			sub	(ix+SP_BESTLAP+1)	;best lap so far?
			jr	c,main_setnewbest	;Yes: go set it
			jr	main_nofirstlap		;No: don't set it

main_firstlap:		ld	bc,(main_ticks)		;If it's the first lap, it will also be the best
main_setnewbest:	ld	(ix+SP_BESTLAP+0),b	;Store best lap so far
			ld	(ix+SP_BESTLAP+1),c

main_nofirstlap:	ld	a,l			;Get number of laps
			ld	hl,current_stage_laps
			cp	(hl)			;Check if we've completed all laps
			jp	nz,main_restart		;No: Continue game
							;Yes:

			set	2,(ix+SP_STATE)		;Mark winning state in car
			ld	hl,tableadvance
			inc	(hl)
			ld	a,(tablepointer)
			ld	l,a
			ld	a,(hl)
			ld	(ix+SP_MAPPOINTS),a 	;Points for this stage.
			
			bit	7,(ix+SP_STATE)		;Human?
			jr	z,main_finishednothuman
			ld	a,(humansarrived)	;If human, increase the number of finished humans
			inc	a
			ld	(humansarrived),a
			set	6,(ix+SP_STATE)		;Mark the we've arrived bit
main_finishednothuman:	ld	a,(positionwin)
			inc	a
			ld	(positionwin),a		;Increase the winning position
			ld	d,a
			ld	a,(nactive_cars)
			cp	d			;Everybody has arrived?
			jp	nz,main_nocheck		;No: Continue game
							;Yes
			call	preparemarker		;Update marker
			call	printthemarker		;Print it

			ld	c,0
main_outwaitres:	ld	b,16
main_inwaitres: 	push	bc
			ld	bc,message_presskey
			call	printmessage		;Print "PRESS KEY" message
			pop	bc
			halt				;Wait a frame
			ld	a,(int_ticks)		;Get interrupt tick counter
			ld	hl,msxHZ
			cp	(hl)			;Compare with current msxHZ variable
			jr	nz,main_noticks
			xor	a			;Reset int_ticks every 5/6 frames while the end message is showing
			ld	(int_ticks),a
main_noticks:		call	check_key		;Check for key
			jp	nz,end_stage		;If pressed, go to end_stage
			dec	b
			jr	nz,main_inwaitres	;Loop a bit
			inc	c
			ld	a,c
			and	3
			ld	c,a			;Cycle values 0-3 in register C
			rrca
			jr	c,main_centerit		;If bit 0 is 1, center characters
			jr	z,main_leftit		;If C=0, characters to the left
			call	pmess_select_right	;If C=2, characters to the right
			jr	main_outwaitres
main_centerit:		call	pmess_select_center
			jr	main_outwaitres
main_leftit:		call	pmess_select_left
			jr	main_outwaitres


							;main_cars loop continues here
main_restart:		ld	bc,(main_ticks)
			ld	(ix+SP_LAPSTART+0),b
			ld	(ix+SP_LAPSTART+1),c	;Keep the start time for the next lap
	
main_nocheck:		ld	de,car2-car1
			add	ix,de			;IX points to the next car
			pop	bc			;Restore BC
			dec	b			;If there are more cars left
			jp	nz,main_cars		;Continue loop
							;End main_cars loop
			ret

;**************
;timeprint_car: Print the current lap time for the car, or the best lap time if it has arrived.
;**************
timeprint_car:		bit	2,(ix+SP_STATE)		;Has the car arrived?
			jr	z,tp_current
			ld	h,(ix+SP_BESTLAP)	;Yes: Get the best lap time
			ld	l,(ix+SP_BESTLAP+1)
			jr	timeprint		;     and print it
tp_current:		ld	hl,(main_ticks)		;Get current ticks
			ld	d,(ix+SP_LAPSTART)	
			ld	e,(ix+SP_LAPSTART+1)	;Get lap start timestamp
			and	a
			sbc	hl,de			;Do the substraction


;**********			
;timeprint: Print the timestamp in HL into the camera pointed by IY
;**********
timeprint:		ld	a,15			;0 character minus 1
			ld	de,-6000		;First check the 10 minutes
tp_getnum3:		inc	a			;Increase char
			add	hl,de			;Substract 6000
			jr	c,tp_getnum3		;Keep doing it until we're under 0
			sbc	hl,de			;Add the last 6000 back, so we're over 0 again
			ld	(iy+CAM_NUMMINDEC),a	;Set first digit into the message
			ld	a,2			;Next, 0 character with semicolon minus 1
			ld	de,-600			;Next the minutes
tp_getnum2:		inc	a			;Increase char
			add	hl,de			;Substract 600
			jr	c,tp_getnum2		;Keep doing it until we're under 0
			sbc	hl,de			;Add the last 600 back, so we're over 0 again
			ld	(iy+CAM_NUMMINUNI),a	;Set second digit into the message
			ld	a,15			;Repeat the same process for 10 seconds
			ld	de,-100
tp_getnum1:		inc	a
			add	hl,de
			jr	c,tp_getnum1
			sbc	hl,de
			ld	(iy+CAM_NUMSEGDEC),a
			ld	a,15			;And finally for seconds too
			ld	e,-10
tp_getnum0:		inc	a
			add	hl,de
			jr	c,tp_getnum0
			ld	(iy+CAM_NUMSEGUNI),a
			ret				;No decimals!

;***********
;timeprint3: Print the timestamp in HL into mes_mt_m10 message, with decimals. If the first cipher is a 0, put a space.
;***********
timeprint3:		and	a
			ld	a,15			;character 0 minus 1
			ld	bc,-6000		;10 minutes
tp3_getnum3:		inc	a			;Increase char
			add	hl,bc			;Substract until we're under 0
			jr	c,tp3_getnum3
			sbc	hl,bc			;Restore last
			cp	16
			jr	nz,tp3_firstone
			ld	a,CHR_SPACE		;Replace 0 with a space
tp3_firstone:		ld	(mes_mt_m10),a		;Set char
			jr	tp3_getnum2		;Next cipher

;***********
;timeprint2: Print the timestamp in HL into mes_mt_m10 message, with decimals. If above 10 minutes, display '------', otherwise the first character will be the plus sign.
;***********
timeprint2:		ld	a,CHR_PLUS
			ld	(mes_mt_m10),a		;Set plus sign for first char
			ld	bc,6000
			and	a			;No carry
			sbc	hl,bc			;Substract 6000
			jr	nc,tp2_toomuch		;If it's too much, jump
			add	hl,bc			;Restore time
tp3_getnum2:		ld	a,2			;0 character with semicolon minus 1
			ld	bc,-600			;1 minute
tp2_getnum2:		inc	a			;Increase character
			add	hl,bc
			jr	c,tp2_getnum2		;Do it until we're under 0
			sbc	hl,bc			;Restore las substraction
			ld	(mes_mt_m1),a		;Store minutes
			ld	a,15
			ld	bc,-100			;Repeat for 10 seconds
tp2_getnum1:		inc	a
			add	hl,bc
			jr	c,tp2_getnum1
			sbc	hl,bc
			ld	(mes_mt_s10),a		;Store tens of seconds
			ld	a,15
			ld	c,-10			;Repeat for 1 second
tp2_getnum0:		inc	a
			add	hl,bc
			jr	c,tp2_getnum0
			ld	(mes_mt_s1),a		;Store seconds
			ld	a,26			;Finally, the decimals.
			add	a,l			;Adjust character and add back the last 10 substracted.
			ld	(mes_mt_d),a
			ld	hl,mes_mt_m10		;HL points to message to be written
			ret
tp2_toomuch:		ld	hl,mes_mt_toomuch	;HL now points to "------" message
			ret

;--------------------------------------------------------------------------------------------------------------------------
;Following MSX code by Ramones/ TMHB, taken from http://msxgamesbox.com/karoshi/index.php?topic=628.0 , not commented by me
;************
;search_slot: Looks for our ROM slot. 
;************
search_slot:            call    RSLREG
                        rrca
                        rrca
                        and     3
                        ld      c,a
                        ld      b,0
                        ld      hl, EXPTBL
                        add     hl,bc
                        ld      a,(hl)
                        and     0x80
                        or      c
                        ld      c,a
                        inc     hl
                        inc     hl
                        inc     hl
                        inc     hl
                        ld      a,(hl)
                        and     0x0C
                        or      c;
                        ld      h,0x80
                        ld      (slotvar),a
                        ret

;************
;setrompage0: Sets our ROM in page 0
;************
setrompage0:            ld      a,(slotvar)
                        jr      setslotpage0

;********
;recbios: Position the BIOS ROM
;********
recbios:                ld      a,(EXPTBL)
setslotpage0:           ;di
                        ld      b,a ; B = Slot param in FxxxSSPP format
                        in      a,(0xA8)
                        and     11111100b
                        ld      d,a ; D = Primary slot value    
                        ld      a,b
                        and     3
                        or      d
                        ld      d,a ; D = Final Value for primary slot

                        ; Check if expanded
                        ld      a,b
                        bit     7,a
                        jr      z,recbiosprimary ; Not Expanded
                        
                        and     3
                        rrca
                        rrca
                        and     11000000b
                        ld      c,a
                        ld      a,d
                        and     00111111b
                        or      c
                        ld      c,a ; Primary slot value with main slot in page 3

                        ld      a,b
                        and     00001100b
                        rrca
                        rrca
                        and     3
                        ld      b,a ; B = Expanded slot in page 3
                        ld      a,c
                        out     (0xA8),a ; Slot : Main Slot, xx, xx, Main slot
                        ld      a,(0xFFFF)
                        cpl
                        and     11111100b
                        or      b
                        ld      (0xFFFF),a ; Expanded slot selected
        
recbiosprimary:         ld      a,d ; A = Final value
                        out     (0xA8),a
; Slot Final. Ram, rom c, rom c, Main
                        ret 
;

;End of Ramones' code
;--------------------------------------------------------------------------------------------------------------------------


;***********
;rec_getbit: get one bit from recording stream
;***********
rec_getbit:		ld	a,(current_rec_byte)	;current byte
			add	a,a			;get one bit
			jr	nz,rec_bitgotten
							;If we've read the last bit, it was the marker, so read the next byte
			push	hl
			di
			rst	0			;Set RAM in page 2

			ld	hl,(current_rec_addr)	;Get next byte address
			ld	a,(hl)			;Get byte
			inc	hl
			ld	(current_rec_addr),hl	;Update address
			pop	hl
			rst	8			;Set ROM in page 2
			scf				;Marker bit
			rla				;Get first bit
rec_bitgotten:		ld	(current_rec_byte),a	;Update current byte
			ret

;*************
;rec_get4bits: Get 4 bits in lower register C from record stream
;*************
rec_get4bits:		ld	b,4			;4 bits
rec_getting4:		call	rec_getbit		;Get 1 bit in carry flag
			ld	a,c
			rra				;Enter bit into register from the right
			ld	c,a
			djnz	rec_getting4		;Repeat 4 times
			rrca
			rrca
			rrca
			rrca				;Move bits to low positions
			and	15			;Clear high bits
			ld	c,a			;Result in C
			ret

;************
;setrec4bits: Insert the 4 lower bits of register C into the recording stream
;************
setrec4bits:		ld	a,(recording)
			and	a
			ret	z			;Return if 16k mode
			push	bc
			ld	b,4			;Number of bits to enter
setrecloop:		rr	c			;Get bit
			ld	a,(current_rec_byte)	;Get current byte
			adc	a,a			;Add bit to current byte
			jr	nc,setrecloopout
							;If carry, we need to store this byte.
			rst	0			;Set RAM in page 2
			di				;Disable interrupts
			push	hl
			ld	hl,(current_rec_addr)	;Get current recording address
			ld	(hl),a			;Store byte
			inc	hl			;Increase address
			ld	(current_rec_addr),hl	;Store it
			pop	hl
			rst	8			;Set ROM in page 2
			ld	a,1			;Marker bit
setrecloopout:		ld	(current_rec_byte),a	;Update current byte
			djnz	setrecloop		;Repeat for all bits
			pop	bc
			ret

;********
;setrec1: Insert a 1 bit into the recording stream
;********
setrec1:		scf				;Set carry flag
			jr	setrec

;********
;setrec0: Insert a 0 bit into the recording stream
;********
setrec0:		and	a			;Clear carry flag
setrec:			push	af			;Keep flags in stack
			ld	a,(recording)
			and	a
			jr	z,setrec_cleanexit	;Go to return if we're in 16k mode
			pop	af			;Get flag
setrecc:		ld	a,(current_rec_byte)	;Get current byte
			adc	a,a			;Add bit
			jr	nc,setrecend
			push	af			;If carry, we need to store this byte.
			rst	0			;Set RAM in page 2
			di				;Disable interrupts
			pop	af
			push	hl
			ld	hl,(current_rec_addr)	;Get current recording address
			ld	(hl),a			;Store byte
			inc	hl			;Increase address
			ld	(current_rec_addr),hl	;Store it
			pop	hl
			rst	8			;Set ROM in page 2
			ld	a,1
setrecend:		ld	(current_rec_byte),a	;Update current byte
			ret
setrec_cleanexit:	pop	af			;Discard flags from stack
			ret

;*************
;record_flush: End recording by flushing the needed bits to complete one last byte, and store final cars state
;*************
record_flush:		ld	a,(recording)		;Flush the last bits
			cp	1
			ret	nz			;Return if we're replaying
			ld	a,(current_rec_byte)	;Get current byte
record_flushing:	add	a,a			;Flush
			jr	nc,record_flushing	;Keep moving bits to the left until the marker bit gets out
			ld	hl,(current_rec_addr)	;Get recording address
			ld	b,a			;Keep byte in B
			rst	0			;Set RAM in page 2
			ld	(hl),b			;Store last byte
			ld	(max_rec_addr),hl	;Store ending position of recording
			ld	hl,car1			;Copy from here
			ld	de,32768		;To here
			ld	bc,end_cars-car1	;Copy final stage state
			ldir				;Copy final stage state
			jp	8			;instead of rst 8 and ret

;*********************
;record_preparerecord: Prepare to record
;*********************
record_preparerecord:	ld	hl,32768+end_cars-car1
			ld	(current_rec_addr),hl	;Set start address
			ld	a,1
			ld	(current_rec_byte),a	;Set first marker bit
			ret

;*******************
;record_prepareplay: Prepare the replay
;*******************
record_prepareplay:	ld	hl,32768+end_cars-car1
			ld	(current_rec_addr),hl	;Set start address
			ld	a,128
			ld	(current_rec_byte),a	;Set first marker bit
			ld	a,2
			ld	(recording),a		;Recording mode to 2 (replay)
			ld	ix,car1			;First car
			ld	b,4			;Number of cars
record_writecontrol:	ld	de,control_play
			ld	(ix+SP_CONTROL+1),d
			ld	(ix+SP_CONTROL),e	;For every car, set their control to control_play
			ld	de,car2-car1
			add	ix,de			;Point to next car
			djnz	record_writecontrol
			ret

;*********
;doreplay: Start replay of previous race
;*********
doreplay:		call	record_prepareplay	;Prepare the replay
			ld	hl,endreplay
			ld	(end_stage_redirect),hl	;Redirect end_stage routine
			jp	start_stage		;Start!

;**********
;endreplay: End replay of race
;**********
endreplay:		ld	hl,end_stage_normal
			ld	(end_stage_redirect),hl	;Restore end_stage routine for the next time
			ld	a,1
			ld	(recording),a		;Recording variable at 1
			rst	0			;Set RAM in page 2
			ld	hl,32768
			ld	de,car1
			ld	bc,end_cars-car1
			ldir				;Copy all cars data
			rst	8			;Set ROM in page 2
			jp	end_stage_reentry	;Go back to end stage interlude

;**********
;end_stage: Do after playing calculations and show interlude
;**********
end_stage:		call	wait_nokey		;Wait for no key to be pressed
			ld	hl,(end_stage_redirect)	;Get redirection
			jp	(hl)			;Go to redirected end_stage

end_stage_normal:	call	record_flush		;If we were recording, flush the record
end_stage_reentry:	call	silencia_ay		;Silence the PSG chip
			call	cls			;Clear screen
			ld	hl,screeninter
			ld	de,screenbuffer
			call	deexo			;Uncompress interlude screen
			call	minimap_center
			ld	hl,1800h
			ld	bc,768
			ld	de,screenbuffer
			call	ldrvrm			;Interlude ready...
			ld	hl,800h+8*132
			ld	bc,57*8
			call	ldrvrmbuffer		;Print map
			ld	a,4
			call	CARGA_CANCION		;Load interlude song
			di				;Disable interrupts
			ld	hl,isr_wyz
			ld	(interrupt_redir),hl	;Set WYZPlayer interrupt routine
			ei				;Music!
			call	pmess_select_center	;Select centered text

			ld      bc,tr_trackres_fixed
			call	printmessage		;Print fixed interlude message
			
			ld	b,4			;Number or cars
			ld	ix,car1			;First car
			ld	h,255			;Max time for best lap before any has been set
esta_prevloop:		push	bc
			bit	4,(ix+SP_STATE)
			jr	nz,esta_prevjumpcar	;Don't check if car isn't active
			ld	a,(ix+SP_BESTLAP)
			and	a
			jr	z,esta_prevjumpcar	;if zero, we're just aborting!
			cp	h			;For each car that didn'd abort, check if the best lap is better than the one we already had
			jr	c,esta_newbestlap
			jr	nz,esta_prevjumpcar
			ld	a,(ix+SP_BESTLAP+1)
			cp	l
			jr	nc,esta_prevjumpcar
esta_newbestlap:	ld	h,(ix+SP_BESTLAP)	;Best lap found so far in HL
			ld	l,(ix+SP_BESTLAP+1)
esta_prevjumpcar:	ld	bc,car2-car1
			add	ix,bc
			pop	bc
			djnz	esta_prevloop		;Loop with all cars
			ld	(bestlap),hl		;Store best lap

			ld	b,4
			ld	ix,car1
			ld	e,16			;start address
			ex	af,af'
			xor	a			;Clear A'
			ex	af,af'
esta_extloop:		ex	af,af'
			and	a			;Clear Carry in F'
			ex	af,af'
			push	bc
			bit	4,(ix+SP_STATE)
			jp	nz,esta_jumpcar		;Don't check if the car didn't run
			ld	a,(ix+SP_COLORMARKER)	;Get color
			ld	(mes_mt_color),a	;Set color for timestamp message
			sub	3
			ld	(mes_points_col),a	;Set color for points message (3 less characters)
			ld	a,e
			ld	(mes_mt_position+1),a	;Set position for timestamp message
			ld	h,(ix+SP_TIMESTAMP)
			ld	l,(ix+SP_TIMESTAMP+1)	;Get timestamp (total time)
			call	timeprint3		;Print it in timestamp message
			push	de			;Keep position
			ld	bc,mes_mt_position
			call	printmessage		;Print total time timestamp
			pop	de			;Restore position
			ld	a,6
			add	a,e			
			ld	(mes_mt_position+1),a	;Update position for next timestamp
			ld	e,a
			ld	h,(ix+SP_BESTLAP)	;Get timestamp (best lap)
			ld	l,(ix+SP_BESTLAP+1)
			push	hl			;Keep it
			call	timeprint3		;Print it in timestamp message
			push	de			;Keep position
			ld	bc,mes_mt_position	
			call	printmessage		;Print best lap timestamp
			pop	de			;Restore position
			ld	a,6
			add	a,e
			ld	e,a			;Update position for points
			ld	a,(ix+SP_MAPPOINTS)
			add	a,16
			ld	(mes_points),a		;Set points in points message
			pop	bc			;Get best lap for this car
			ld	hl,(bestlap)
			sbc	hl,bc			;Compare with total best lap
			jr	nz,esta_nobestlap
			ld	a,CHR_PLUS		;If we have the best lap, add the extra points
			ld	(mes_extrapoints),a	;Set plus sign
			ld	a,(tablepointsbl)	
			add	a,16
			ld	(mes_extrapoints+1),a	;Set the best lap points
			ld	a,(tablepointsbl)
			add	a,(ix+SP_POINTS)
			ld	(ix+SP_POINTS),a	;Add to total points of this track
			ex	af,af'
			scf				;Set carry flag in F'
			ex	af,af'
			jr	esta_printpoints
esta_nobestlap:		ld	a,CHR_SPACE		;If we don't have the best lap
			ld	(mes_extrapoints),a
			ld	(mes_extrapoints+1),a	;Set spaces in the last two chars
esta_printpoints:	push	de			;Keep position
			ld	bc,mes_points_col
			call	printmessage		;Print points message
			pop	de			;Restore position
			ld	a,3
			add	a,e
			ld	e,a			;Update position for next line
			ld	a,(ix+SP_MAPPOINTS)
			add	a,(ix+SP_POINTS)
			ld	(ix+SP_POINTS),a	;Update total points with points given by position in track
esta_jumpcar:		ex	af,af'			;A' (now A) has the records byte
			rla				;Enter bit telling if it has the record or not
			bit	7,(ix+SP_STATE)		;Check human bit
			jr	z,esta_nothuman
			add	a,16			;Set human bit if needed
esta_nothuman:		ex	af,af'			;Keep the records byte in A' again
			ld	bc,car2-car1
			add	ix,bc			;Point to next car
			pop	bc
			dec	b
			jp	nz,esta_extloop		;Loop for all cars

			ld	a,(currentmap)		;Get current map
			add	a,a
			add	a,a
			add	a,a
			add	a,a			;*16
			ld	e,a
			ld	a,(mode)
			add	a,e			;Add category
			ld	e,a
			ld	d,0
			ld	ix,mapas+2
			ld	iy,mapas+6
			add	ix,de			;0 -> points to record holders
			ld	a,(mode)
			add	a,e			;Add category again
			ld	e,a
			add	iy,de			;0,1 -> points to record

			ld	d,(iy+0)
			ld	e,(iy+1)		;DE=old record
			ld	hl,(bestlap)
;			and	a			;No carry since the last "add iy,de"
			sbc	hl,de
			jr	nc,esta_nonewrecord
			add	hl,de			;A new record has been made
			ld	(iy+0),h
			ld	(iy+1),l		;Set new record
			ex	af,af'
			ld	(ix+0),a		;Set new record holders

esta_nonewrecord:	ld	ix,car1			;IX points to first car
			ld	c,4			;4 cars
			ld	de,car2-car1
esta_outerposition:	ld	b,4			;4 cars, initial position
			ld	(ix+SP_MAPPOINTS),b	;Set position 4 for IX's car
			ld	iy,car1			;IY points to first car
esta_innerposition:	ld	a,(ix+SP_POINTS)
			cp	(iy+SP_POINTS)		;Compare points for next car
			jr	c,esta_nodecpos
			dec	(ix+SP_MAPPOINTS)	;Decrease position if the other car has more points
esta_nodecpos:		add	iy,de			;IY points to next car
			djnz	esta_innerposition	;Loop all cars pointed by IY
			add	ix,de			;IX points to next car
			dec	c
			jr	nz,esta_outerposition	;Loop all cars

			ld	hl,tr_col1		;Position of 1st classified (so far) color code
			ld	c,0			;Position to look for
esta_outerposition2:	ld	b,4			;Number of cars
			ld	ix,car1			;Point to first car
esta_innerposition2:	push	bc
			ld	a,c
			cp	(ix+SP_MAPPOINTS)		;Check position
			jr	nz,esta_dontprintpointshere	;Don't do anything if it's not the one we're looking for
                        bit     4,(ix+SP_STATE)			;Check if the cars runs
                        jr      nz,esta_dontprintpointshere	;Don't do anything if the car didn't run
			ld	a,(ix+SP_COLORMARKER)	;Get color
			sub	2			;Only 4 repeats
			ld	(hl),a			;Set color in results so far message
			inc	hl			;Point to position
			ld	a,c			
			add	a,17			;Transform position into number character
			ld	(hl),a			;Set position character
			inc	hl
			inc	hl			;Point to points
			ld	a,(ix+SP_POINTS)		
			ld	b,15			;0 character minus 1
esta_mod10:		inc	b			;Increase tens digit
			sub	10			;Substract 10
			jr	nc,esta_mod10		;Keep increasing it while score > 0
			add	a,26			;Revert back the last substraction, and add the constant to form character
			ld	(hl),b			;Store first digit
			inc	hl
			ld	(hl),a			;Store second digit
			inc	hl			;Point to next line
esta_dontprintpointshere:
			pop	bc
			add	ix,de			;Point to next car
			djnz	esta_innerposition2	;Loop for all cars
			inc	c			;Next position to check
                        ld      a,(nactive_cars)
                        cp      c			;Check against number of cars
                        jr      nz,esta_outerposition2	;Loop while there are cars
                        cp      4			;position 4 reached?
                        jr      z,esta_nomoreclass	;Yes: no more lines to process
                        ld	a,131+6*16+4		; No: Black color, 4 chars
                        ld      (tr_col4),a		;     for the fourth line
                        ld      a,c
                        cp      2			;Only two cars
                        jr      nz,esta_nomoreclass	; No: no more lines to process
                        ld	a,131+6*16+4		;Yes: Black color, 4 chars
                        ld      (hl),a			;     for the third line too
esta_nomoreclass:       ld      bc,tr_trackres		;Message for total results so far
			call	printmessage		;Print it
			ld	a,(has32k)
			and	a
			jr	z,esta_16k		;16k?
			ld	bc,message_replay	;No: Message including "R to Replay"
			call	printmessage
			call	waitkey_60		;Wait for key to be pressed, taking 60Hz into consideration
			call	silencia_ay		;Silence the PSG
			ld	a,(inAA)
			or	4			;row R-K
			out	(#AA),a
			in	a,(#A9)			;read row into A
			and	128			;Check key R
			jp	z,doreplay		;If pressed, do replay
			jr	esta_32k
esta_16k:		ld	bc,tr_presskey		;Message without the "R to Replay" part
			call	printmessage
			call	waitkey_60		;Wait for key to be pressed, taking 60Hz into consideration
			call	silencia_ay		;Silence the PSG
esta_32k:		ld	a,(currentmap)
			inc	a
			ld	(currentmap),a		;Update current map
			cp	8
			jp	c,start_stage		;Start next stage if we haven't reached 8

;********
;endgame: Show the end of the game, with the pilots on the podium
;********
endgame:		ld	a,4
			call	CARGA_CANCION		;Load interlude song in WYZPlayer
			ld	hl,menuchars
			call	deexo_buffer		;Uncompress I Need Speed logo characters
			ld	hl,8
			ld	bc,448			;Number of bytes
			call	ldrvrmbuffer		;Logo chars, 1st third
			ld	hl,podium
			ld	de,mapa
			call	deexo			;Uncompress podium screen into mapa buffer
			ld	bc,31*8			;31 characters
			ld	de,mapa
			ld	hl,808h			;
			call	ldrvrm			;Write podium characters into VRAM
			ld	bc,31*8
			ld	de,mapa+31*8
			ld	hl,2808h
			call	ldrvrm			;Write podium colors into VRAM

			ld	hl,mapa+31*16
			ld	de,screenbuffer
			ld	bc,768
			ldir				;Write podium screen into VRAM

			ld	hl,endg_n1
			ld	de,car2-car1
			ld	c,0			;Test for number 1
endg_loopcount_out:	ld	b,4			;Number of cars
			xor	a
			ld	(hl),a			;Reset number of cars found with the testing position
			ld	ix,car1			;First car
endg_loopcount_inn:	bit	4,(ix+SP_STATE)
			jp	nz,endg_loopcount_end	;Don't check if the car doesn't race
			ld	a,c
			cp	(ix+SP_MAPPOINTS)	;MAPPOINTS holds the position
			jr	nz,endg_loopcount_end
			inc	(hl)			;If found, increase number of occurences
endg_loopcount_end:	add	ix,de			;Point to next car
			djnz	endg_loopcount_inn
			inc	c			;Next position to be checked
			inc	hl
			inc	hl			;Point to next podium position
			ld	a,c
			cp	3
			jr	nz,endg_loopcount_out	;Keep looping until we reach position 3

			ld	c,0			;Test for number 1
			ld	iy,endg_n1		;Number one position and number of occurences
			
endg_loopprint_out:	ld	b,4			;Number of cars
			ld	a,(iy+1)		;Get screen position for this ranking
			sub	(iy)			;Sub number of pilots with this ranking
			ld	l,a			;L = position
			ld	ix,car1			;Start looking car 1
endg_loopprint_inn:	bit	4,(ix+SP_STATE)
			jp	nz,endg_loopprint_end	;Don't check if the car doesn't race
			ld	a,c			;Check if position
			cp	(ix+SP_MAPPOINTS)	;is the one we're looking for
			jr	nz,endg_loopprint_end
			push	hl			;Store initial position
			ld	de,31
			ld	h,(screenbuffer/256)+1	;2nd third high part, now HL has the position
			ld	a,1
			ld	(hl),a			;Set first head byte
			inc	hl			;Next character position
			inc	a
			ld	(hl),a			;Set second head byte
			add	hl,de			;Next line position
			inc	a
			ld	(hl),a			;Set third head byte
			inc	hl			;Next character position
			inc	a
			ld	(hl),a			;Head ready
			add	hl,de			;Next line position
			ld	a,8
			sub	b			;A = 8 - B, B goes from 4 (for car1) to 1 (for car 4)
			add	a,a
			add	a,a			;Multiply by 4, getting the initial body byte for current pilot
			ld	(hl),a			;Set first body byte
			inc	hl			;Next character position
			inc	a
			ld	(hl),a			;Set second body byte
			add	hl,de			;Next line position
			inc	a
			ld	(hl),a			;Set third body line
			inc	hl			;Next character position
			inc	a
			ld	(hl),a			;Body ready
			pop	hl			;Recover initial position
			inc	l
			inc	l			;Next pilot will be 2 characters to the right
endg_loopprint_end:	ld	de,car2-car1
			add	ix,de			;Point to next car
			djnz	endg_loopprint_inn	;Keep printing all pilots
			inc	c			;Next position to be checked
			inc	iy
			inc	iy			;Point to next position
			ld	a,c
			cp	3
			jr	nz,endg_loopprint_out	;Keep looping until we reach position 3

			ld	hl,1800h
			ld	bc,768
			ld	de,screenbuffer
			call	ldrvrm			;Dump the screen to VRAM
			ld	bc,end_podium
			call	printmessage		;Print the podium message
			call	vdpena			;Enable VDP

			call	micropause		;Do a small pause
			call	wait_nokey		;Wait for no keys to be pressed
			call	wait_key		;Wait for a key to be pressed
			call	silencia_ay		;Silence the PSG
			call	checkgirl		;Call the special image routine
			jp	menu			;Back to menu

;***************
;printthemarker: Print the marker
;***************
printthemarker:		ld	bc,marker_message	;Marker message
printthemarker2:	call	printmessage		;Print it
			xor	a
			ld	(changemarker),a	;No need to print it again next main loop
			ret

;************
;silencia_ay: Silence the PSG chip
;************
silencia_ay:		di				;Disable interrupts
			call	SILENCIAPLAYER		;Silence the player
			ld	hl,int_end
			ld	(interrupt_redir),hl	;No extra process after interrupt
			ei				;Enable interrupts
			ret

;****
;cls: Clear screen
;****
cls:			xor	a			;0 value
			ld	hl,buffer		;Buffer
			ld	de,buffer+1
			ld	bc,2047			;Number of bytes to fill
			ld	(hl),a			;Set 0 in first value of buffer
			ldir				;Fill Buffer with zeros
			ld	hl,0			;VRAM address
			call	ldrvrmbuffer2k		;Copy 2k of zeros to first third characters
			ld	hl,800h			;VRAM address
			call	ldrvrmbuffer2k		;Copy 2k of zeros to second third characters
			ld	hl,1000h		;VRAM address
			call	ldrvrmbuffer2k		;Copy 2k of zeros to third third characters
			ld	hl,2000h		;VRAM address
			call	ldrvrmbuffer2k		;Copy 2k of zeros to first third colors
			ld	hl,2800h		;VRAM address
			call	ldrvrmbuffer2k		;Copy 2k of zeros to second third colors
			ld	hl,3000h		;VRAM address
			call	ldrvrmbuffer2k		;Copy 2k of zeros to third third colors
			ld	hl,1b00h		;VRAM address
			ld	bc,128			;Number of bytes
			jp	ldrvrmbuffer

;******************
;menuprintcontrols: Prints the control option for each car
;******************
menuprintcontrols:	call	pmess_select_center	;Centered font
			ld	b,4
			ld	ix,car1			;Start with the first car
			ld	hl,74*8			;First third, character 74 VRAM address
menpr_popti:		ld	(pmess_address),hl	;Set VRAM address for printmessage routine
			push	bc
			push	hl
			ld	b,(ix+SP_CONTOPT)	;Get control option for the car
			ld	de,11
			ld	hl,mmenu_keys1-11	;First option address-11
			inc	b
menpr_getadd:		add	hl,de			;Next option address
			djnz	menpr_getadd
							;Here HL points to the option
			ld	b,h
			ld	c,l			;BC = HL
			call	printmessage_nr		;Print car's control
			ld	bc,car2-car1
			add	ix,bc			;Next car
			ld	bc,13*8
			pop	hl
			add	hl,bc			;VRAM address for next car message
			pop	bc
			djnz	menpr_popti

			ld	a,(mode)		;Get category
			ld	c,a
			add	a,a
			add	a,a
			add	a,a			;*8
			sub	c			
			ld	c,a			;C=category*7
			ld	hl,800h+39*8		;VRAM address for character 39 in second third
			ld	(pmess_address),hl	;Set it for printmessage
			ld	hl,mmenu_cat0		;Categories
			add	hl,bc			;Add category*7
			ld	b,h
			ld	c,l			;BC = HL
			call	printmessage_nr		;Print category
			
			ld	a,(championship)	;Get championship
			ld	e,a
			add	a,a
			add	a,a
			add	a,a			;*8
			ld	d,0
			add	a,e			;*9
			ld	e,a
			ld	hl,mmenu_champ0		;Championships
			add	hl,de			;Add championship*9
			ld	b,h
			ld	c,l			;BC = HL
			call	printmessage_nr		;Print championship
			ld	hl,(mmenu_cam1shadow)
			ld	(mmenu_keycam1),hl	;Copy cam1shadow values to visible ones
			ld	bc,mmenu_extrakeys
			jp	printmessage_nr		;Print extra keys, then return

;***************
;testlegal_chkz: Check how many car controls have a certain value
;***************
testlegal_chkz:		ld	l,0		;Counter to 0
			cp	b
			jr	nz,tl_nob
			inc	l		;Increase if car 1 has it
tl_nob:			cp	c
			jr	nz,tl_noc
			inc	l		;Increase if car 2 has it
tl_noc:			cp	d
			jr	nz,tl_nod
			inc	l		;Increase if car 3 has it
tl_nod:			cp	e
			ret	nz
			inc	l		;Increase if car 4 has it
			ret

;***************
;testlegal_chkc: Check how many car controls are below a certain value
;***************
testlegal_chkc:		ld	l,0		;Counter to 0
			cp	b
			jr	c,tl_nobc
			inc	l		;Increase if car 1 is below
tl_nobc:		cp	c
			jr	c,tl_nocc
			inc	l		;Increase if car 2 is below
tl_nocc:		cp	d
			jr	c,tl_nodc
			inc	l		;Increase if car 3 is below
tl_nodc:		cp	e
			ret	c		
			inc	l		;Increase if car 4 is below
			ret

;***************
;menu_testlegal: Check if a certain car control option is legal or not, depending on the number of cars already controlled by humans or that don't race
;***************
menu_testlegal:		ld	a,(car1+SP_CONTOPT)
			ld	b,a			;B = Control option for car 1
			ld	a,(car2+SP_CONTOPT)
			ld	c,a			;C = Control option for car 2
			ld	a,(car3+SP_CONTOPT)
			ld	d,a			;D = Control option for car 3
			ld	a,(car4+SP_CONTOPT)
			ld	e,a			;E = Control option for car 4
			ld	a,9
			call	testlegal_chkz		;Check how many cars have the option 9 (don't race)
			ld	a,2
			cp	l
			ret	c			;If above 2, exit with carry (illegal)
			ld	a,5
			call	testlegal_chkc		;Check how many cars have the option below 5 (human)
			ld	a,2
			cp	l
			ret	c			;If above 2, exit with carry (illegal)
			ret	nz			;If not 2, exit without carry (legal)

			ld	h,0			;Start with 0
tl_nocollis:		ld	a,h
			call	testlegal_chkz		;Check how many cars have option A
			ld	a,1
			cp	l
			ret	c			;If above 1, exit with carry (illegal) -> both cars have the same control
			ld	a,h
			inc	a
			ld	h,a			;Next value
			cp	6
			jr	nz,tl_nocollis		;Keep trying from 0 to 5
			ret


			INCLUDE	"pt3player.asm"		;Sapphire's PT3 Player to play menu music
girlch2:		INCBIN	"bindata/girlch2.exp"	;Girl picture characters, third 2
girlco01:		INCBIN	"bindata/girlco01.exp"	;Girl picture colors, thirds 0 and 1
checkinram:		defb	55
girlco2:		INCBIN	"bindata/girlco2.exp"	;Girl picture colors, third 2
free1:
			ORG	32768
			INCBIN	"ins2.bin"		;Page 2 data
varpacked:		INCBIN	"varpack.exp"		;Packed initial variables at the end of page 2
free2:
			ORG	49151
			defb	0			;Make it a 32k ROM

zz_MEMFREE0		EQU	04000h-free0				;Free memory in page 0
zz_MEMFREE1		EQU	08000h-free1				;Free memory in page 1
zz_MEMFREE2		EQU	0C000h-free2				;Free memory in page 2
zz_MEMFREE_TOTAL	EQU	zz_MEMFREE0+zz_MEMFREE1+zz_MEMFREE2	;Total free memory